import {ActionType, getType} from 'typesafe-actions';
import {call, put, select} from 'redux-saga/effects';
import omit from 'lodash/omit';
import {SagaIterator} from 'redux-saga';

import {URLs} from 'constants/urls';

import {EAviaActionLogActionName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogActionName';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {TAviaActionLogActionFilters} from 'server/loggers/avia/AviaActionLog/types/TAviaActionLogActionFilters';

import filtersReducer, {
    IAviaSearchResultsFilters,
} from 'reducers/avia/search/results/filters/reducer';
import * as filtersActions from 'reducers/avia/search/results/filters/actions';
import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {getAviaSearchFilters} from 'selectors/avia/search/aviaSearchResultsSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import {getAppliedAviaFilters} from 'projects/avia/lib/search/filters/getAppliedFilters';
import {filterValuesToHash} from 'projects/avia/lib/search/filters/converters';
import history from 'utilities/browserHistory/browserHistory';
import scrollTo from 'utilities/dom/scrollTo';
import {reachGoal} from 'utilities/metrika';
import {logError} from 'utilities/logger/logError';
import {unknownToErrorOrUndefined} from 'utilities/error';

const RESET_ACTIONS: string[] = [
    getType(filtersActions.resetFilterValue),
    getType(filtersActions.resetDirectionAirportsFilter),
    getType(filtersActions.resetTransferAirportsFilter),
    getType(filtersActions.resetTransferFilter),
    getType(filtersActions.resetFilterValues),
];

const ACTIONS_MAP: Record<string, EAviaGoal> = {
    [getType(filtersActions.setBaggageFilter)]: EAviaGoal.SEARCH_BAGGAGE_FILTER,
    [getType(filtersActions.setTransferFilter)]:
        EAviaGoal.SEARCH_TRANSFER_FILTER,
    [getType(filtersActions.setTransferRangeFilter)]:
        EAviaGoal.SEARCH_TRANSFER_RANGE_FILTER,
    [getType(filtersActions.setTimeFilter)]: EAviaGoal.SEARCH_TIME_FILTER,
    [getType(filtersActions.setCompanyFilter)]: EAviaGoal.SEARCH_COMPANY_FILTER,
    [getType(filtersActions.setDirectionAirportsFilter)]:
        EAviaGoal.SEARCH_AIRPORTS_FILTER,
    [getType(filtersActions.setTransferAirportsFilter)]:
        EAviaGoal.SEARCH_AIRPORTS_FILTER,
    [getType(filtersActions.setPriceFilter)]: EAviaGoal.SEARCH_PRICE_FILTER,
    [getType(filtersActions.setPartnersFilter)]:
        EAviaGoal.SEARCH_PARTNERS_FILTER,
};

function getActionFilters(
    action: ActionType<typeof filtersActions>,
    filters: IAviaSearchResultsFilters,
): TAviaActionLogActionFilters {
    switch (action.type) {
        case getType(filtersActions.setBaggageFilter):
            return {
                name: 'baggage',
                value: filters.baggage,
            };
        case getType(filtersActions.setTransferFilter):
        case getType(filtersActions.setTransferRangeFilter):
            return {
                name: 'transfer',
                value: filters.transfer,
            };
        case getType(filtersActions.setTimeFilter):
            return {
                name: 'time',
                value: filters.time,
            };
        case getType(filtersActions.setCompanyFilter):
            return {
                name: 'company',
                value: filters.company,
            };
        case getType(filtersActions.setDirectionAirportsFilter):
            return {
                name: 'airports',
                value: filters.airports,
            };
        case getType(filtersActions.setTransferAirportsFilter):
            return {
                name: 'airports',
                value: filters.airports,
            };
        case getType(filtersActions.setPriceFilter):
            return {
                name: 'price',
                value: filters.price,
            };
        case getType(filtersActions.setPartnersFilter):
            return {
                name: 'partners',
                value: filters.partners,
            };
        case getType(filtersActions.resetFilterValues):
            return {
                name: 'total',
                value: 'reset',
            };
        case getType(filtersActions.resetFilterValue):
            return {
                name: action.payload,
                value: 'reset',
            };
        case getType(filtersActions.resetDirectionAirportsFilter):
            return {
                name: 'airports',
                value: 'reset',
            };
        case getType(filtersActions.resetTransferAirportsFilter):
            return {
                name: 'transfer',
                value: 'reset',
            };
        case getType(filtersActions.resetTransferFilter):
            return {
                name: 'transfer',
                value: 'reset',
            };
        default:
            return {
                name: 'UNKNOWN_ACTION',
            };
    }
}

function* filterChangeHandler(
    actionsData:
        | ActionType<typeof filtersActions>
        | ActionType<typeof filtersActions>[],
): SagaIterator {
    try {
        const actions = Array.isArray(actionsData)
            ? actionsData
            : [actionsData];
        const preparedAction =
            actions.length > 1 &&
            actions.every(
                currentAction =>
                    currentAction.type ===
                    getType(filtersActions.resetFilterValue),
            )
                ? [filtersActions.resetFilterValues()]
                : actions;
        const filters = yield select(getAviaSearchFilters);

        for (const action of preparedAction) {
            const nextState = filtersReducer(filters, action);
            const appliedFilters = getAppliedAviaFilters(nextState);
            const filtersHash = filterValuesToHash(appliedFilters);

            if (!RESET_ACTIONS.includes(action.type)) {
                yield call(reachGoal, EAviaGoal.SEARCH_FILTER);
                yield call(reachGoal, ACTIONS_MAP[action.type]);
            }

            if (history) {
                const {pathname, search, state} = history.location;

                // Изменение хеша имеет смысл только на странице авиапоиска
                if (pathname.includes(URLs.aviaSearchResults)) {
                    yield call(() => {
                        history?.replace({
                            hash: filtersHash,
                            pathname,
                            search,
                            state: state && omit(state, 'modalId'),
                        });
                    });
                }
            }

            yield put(
                loggerActions.logSearchAction({
                    name: EAviaActionLogActionName.SEARCH_PAGE_FILTERING,
                    params: {
                        actionFilters: getActionFilters(action, filters),
                    },
                }),
            );
        }

        scrollTo({top: 0});
    } catch (err) {
        logError(
            {
                message:
                    '[YATRAVEL][AVIA] Возникла проблема при обработке события фильтрации',
                block: 'filtersSaga',
            },
            unknownToErrorOrUndefined(err),
        );
    }
}

export default function* filtersSaga() {
    yield watchActions(
        filterChangeHandler,
        [
            filtersActions.setBaggageFilter,
            filtersActions.setTransferFilter,
            filtersActions.setTransferRangeFilter,
            filtersActions.setTimeFilter,
            filtersActions.setCompanyFilter,
            filtersActions.setDirectionAirportsFilter,
            filtersActions.setTransferAirportsFilter,
            filtersActions.setPriceFilter,
            filtersActions.setPartnersFilter,
            filtersActions.setPlusPointsFilter,
            filtersActions.resetFilterValue,
            filtersActions.resetDirectionAirportsFilter,
            filtersActions.resetTransferAirportsFilter,
            filtersActions.resetTransferFilter,
            filtersActions.resetFilterValues,
        ].map(getType),
        true,
    );
}
