import {
    all,
    call,
    CallEffect,
    cancel,
    delay,
    fork,
    put,
    race,
    select,
    take,
    takeLatest,
} from 'redux-saga/effects';
import {SagaIterator, Task} from 'redux-saga';
import {ActionType, getType} from 'typesafe-actions';
import {assign, isUndefined, omitBy} from 'lodash';

import {MINUTE} from 'utilities/dateUtils/constants';
import {
    SOLOMON_AVIA_SEARCH_FAILED,
    SOLOMON_AVIA_SEARCH_FAILED_WRONGSTART,
    SOLOMON_AVIA_SEARCH_INIT,
    SOLOMON_AVIA_SEARCH_INIT_ERROR_BACKEND,
    SOLOMON_AVIA_SEARCH_INIT_ERROR_REQUEST,
    SOLOMON_AVIA_SEARCH_SUCCESS_WITH_OFFERS,
    SOLOMON_AVIA_SEARCH_SUCCESS_WITHOUT_OFFERS,
} from 'constants/solomon/avia';

import {
    EAviaTDAnswerStatusCode,
    IAviaTDAnswer,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {IAviaDataByPointKey} from 'server/services/AviaGeoService/types/IAviaDataByPointKey';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {TPromiseReturnType} from 'types/utilities';
import {IGetExtendedCountryRestrictionsResponse} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';

import {
    fetchPartnersInfo,
    initSearch,
    initSearchFail,
    initSearchSuccess,
    setPartnersInfo,
    setTDAnswer,
    stopSearch,
    updateAviaSearchProgress,
} from 'reducers/avia/search/results/actions';
import {setAviaContext} from 'reducers/avia/context/actions';
import fillFormFromContext from 'reducers/avia/context/fillFormFromContext';
import {setAviaPointsData} from 'reducers/avia/pointsData/actions';
import {IAviaContext, IAviaPointsData} from 'reducers/avia/data-types';
import {StoreInterface} from 'reducers/storeTypes';
import {getCountryRestrictionsActions} from 'reducers/avia/countryRestrictions/actions';
import {getWeatherForecast} from 'reducers/avia/weatherForecast/thunk';

import {getAviaContext, getAviaPointsData} from 'selectors/avia/aviaSelectors';
import experimentsSelector from 'selectors/common/experimentsSelector';

import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import requestSearchSuggests from 'projects/avia/utilities/api/requestSearchSuggests';
import {IPreparedAviaSuggestItemWithIsUniqueTitle} from 'server/services/AviaService/utilities/prepareSuggestsResponse';
import {needShowCountryRestrictions} from 'projects/avia/utilities/needShowCountryRestrictions';
import {getInitSearchError} from 'projects/avia/utilities/getInitSearchError';
import {sendCounter} from 'utilities/solomon';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import {isUnknownAxiosError} from 'utilities/error';

import {IRumUiContext} from 'contexts/RumUiContext';

import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

const POLL_RETRY_COUNT = 10;
const POLLING_DELAY = 2000;

const REQUEST_WITHOUT_VARIANTS_LIMIT = 20 * 1000;
const REQUEST_PARTNERS_LIMIT = 2 * MINUTE;

export default function* (): SagaIterator {
    yield takeLatest(getType(initSearch), handleInitSearch);
    yield takeLatest(getType(fetchPartnersInfo), handleFetchPartnersInfo);
}

function* handleFetchPartnersInfo(): SagaIterator {
    try {
        const partnersInfo = yield call(aviaBrowserProvider.getPartnersInfo);

        if (partnersInfo) {
            yield put(setPartnersInfo(partnersInfo));
        }
    } catch (e) {}
}

function* handleInitSearch({
    payload: {query, rumUi},
}: ActionType<typeof initSearch>): SagaIterator {
    let id;

    const {
        aviaWeatherDesktop,
        aviaOnlyWeatherRight,
        aviaHotelsThenWeatherRight,
    } = yield select(experimentsSelector);

    const needToFetchWeather =
        aviaWeatherDesktop ||
        aviaOnlyWeatherRight ||
        aviaHotelsThenWeatherRight;

    try {
        sendCounter(SOLOMON_AVIA_SEARCH_INIT);

        yield call(updateAviaContext, query);
        yield call(updatePointsData, query);
        yield call(getCountryRestrictions);

        if (needToFetchWeather) {
            yield put(yield call(getWeatherForecast));
        }

        const result = yield call(aviaBrowserProvider.initSearch, query);

        id = result.id;
    } catch (ex) {
        if (isUnknownAxiosError(ex)) {
            yield put(initSearchFail(getInitSearchError(ex?.response?.data)));
        }

        if (isUnknownAxiosError(ex) && ex?.response?.status === 400) {
            sendCounter(SOLOMON_AVIA_SEARCH_INIT_ERROR_REQUEST);
        } else {
            sendCounter(SOLOMON_AVIA_SEARCH_INIT_ERROR_BACKEND);
        }

        return;
    }

    yield put(initSearchSuccess({id}));

    yield race([call(pollResults, id, rumUi), take(getType(stopSearch))]);
}

function* updateAviaContext(params: IAviaParams) {
    const {
        avia: {
            searchForm: {from: fromField, to: toField},
        },
    }: StoreInterface = yield select();

    const fromFieldSelectedValue =
        typeof fromField.selectedValue === 'boolean'
            ? undefined
            : fromField.selectedValue;
    const toFieldSelectedValue =
        typeof toField.selectedValue === 'boolean'
            ? undefined
            : toField.selectedValue;

    const fromId = params.fromId || fromFieldSelectedValue?.pointKey;
    const toId = params.toId || toFieldSelectedValue?.pointKey;

    let fromName = params.fromName || fromFieldSelectedValue?.title || '';
    let toName = params.toName || toFieldSelectedValue?.title || '';

    const dataForAviaContext: Partial<IAviaContext> = {};
    const responses: {
        [P in keyof Partial<IAviaContext>]: CallEffect;
    } = {};

    if (
        fromField.inputValue === fromName &&
        fromFieldSelectedValue?.pointKey === fromId
    ) {
        dataForAviaContext.from = fromFieldSelectedValue;
    } else {
        if (
            !params.fromName &&
            (isSettlementKey(params.fromId) || isStationKey(params.fromId))
        ) {
            const data: IAviaDataByPointKey = yield call(
                aviaBrowserProvider.getDataByPointKey,
                params.fromId,
            );

            fromName = data.station?.title || data.settlement?.title || '';
        }

        responses.from = yield call(
            prepareSuggestResponse,
            toName,
            fromName,
            ESearchFormFieldName.FROM,
        );
    }

    if (
        toField.inputValue === toName &&
        toFieldSelectedValue?.pointKey === toId
    ) {
        dataForAviaContext.to = toFieldSelectedValue;
    } else {
        if (
            !params.toName &&
            (isSettlementKey(params.toId) || isStationKey(params.toId))
        ) {
            const data: IAviaDataByPointKey = yield call(
                aviaBrowserProvider.getDataByPointKey,
                params.toId,
            );

            toName = data.station?.title || data.settlement?.title || '';
        }

        responses.to = yield call(
            prepareSuggestResponse,
            fromName,
            toName,
            ESearchFormFieldName.TO,
        );
    }

    const responsesResult: {
        [P in keyof Partial<IAviaContext>]: TPromiseReturnType<
            typeof prepareSuggestResponse
        >;
    } = yield all(responses);

    assign(dataForAviaContext, omitBy(responsesResult, isUndefined));

    const {from, to} = dataForAviaContext;

    // обновляем поисковый контекст
    if (from && to) {
        yield put(
            setAviaContext({
                ...params,
                fromName,
                toName,
                from,
                to,
            }),
        );

        yield put(
            // @ts-ignore
            fillFormFromContext(),
        );
    }
}

function prepareSuggestResponse(
    // eslint-disable-next-line camelcase
    other_point: string,
    query: string,
    field: ESearchFormFieldName.FROM | ESearchFormFieldName.TO,
): Promise<IPreparedAviaSuggestItemWithIsUniqueTitle | undefined> {
    return requestSearchSuggests({
        query,
        field,
        otherPoint: other_point,
    }).then(suggestData => {
        if (suggestData.items.length > 0) {
            return suggestData.items[0];
        }
    });
}

function* updatePointsData(params: IAviaParams): SagaIterator {
    const pointsDataState = getAviaPointsData(yield select());
    const effects: {[Key in keyof IAviaPointsData]?: CallEffect} = {};

    if (
        (!pointsDataState.fromSettlement ||
            pointsDataState.fromSettlement.title !== params.fromName) &&
        (isSettlementKey(params.fromId) || isStationKey(params.fromId))
    ) {
        effects.from = call(
            aviaBrowserProvider.getDataByPointKey,
            params.fromId,
        );
    }

    if (
        (!pointsDataState.toSettlement ||
            pointsDataState.toSettlement.title !== params.toName) &&
        (isSettlementKey(params.toId) || isStationKey(params.toId))
    ) {
        effects.to = call(aviaBrowserProvider.getDataByPointKey, params.toId);
    }

    const pointsData: Partial<IAviaPointsData> = yield all(effects);

    if (Object.keys(pointsData).length > 0) {
        yield put(setAviaPointsData(pointsData));
    }
}

function* stopPollingByTimeout(): SagaIterator {
    yield delay(REQUEST_WITHOUT_VARIANTS_LIMIT);
    yield put(stopSearch());
}

// TODO: реализовать единый механизм получения цен (https://st.yandex-team.ru/TRAVELFRONT-2999)
// eslint-disable-next-line complexity
function* pollResults(id: string, rumUi: IRumUiContext): SagaIterator {
    const initTime = Date.now();
    let emptyTDAnswerCancellationTask: Task | undefined;

    let retriesLeft = POLL_RETRY_COUNT;
    let cont: number | null = 0;
    let firstResultsReceived = false;

    while (cont !== null) {
        if (Date.now() - initTime > REQUEST_PARTNERS_LIMIT) {
            yield put(stopSearch());

            return;
        }

        try {
            const tdAnswer: IAviaTDAnswer = yield call(
                aviaBrowserProvider.getSearchResults,
                id,
                cont,
            );

            switch (tdAnswer.status) {
                case EAviaTDAnswerStatusCode.INTERNAL_ERROR:
                    emptyTDAnswerCancellationTask = yield fork(
                        stopPollingByTimeout,
                    );

                    break;
                case EAviaTDAnswerStatusCode.NO_OFFERS:
                    yield put(stopSearch());

                    return;
                case EAviaTDAnswerStatusCode.WRONG_REQUEST:
                    yield put(stopSearch());

                    return;
                case EAviaTDAnswerStatusCode.OK:
                case EAviaTDAnswerStatusCode.POLL_CONTINUE:
                default:
                    break;
            }

            const tdAnswerIsEmpty = isEmptyTDAnswer(tdAnswer);

            // Если есть ответ со стороны тикет-демона - берём cont из ответа
            // Если ответ не получили - используем последнее значение
            cont = tdAnswer ? tdAnswer.cont : cont;

            if (tdAnswerIsEmpty && cont === null) {
                sendCounter(SOLOMON_AVIA_SEARCH_SUCCESS_WITHOUT_OFFERS);
            }

            if (!tdAnswerIsEmpty || cont === null) {
                if (!firstResultsReceived) {
                    rumUi.measure(EAviaRumEvents.TDAnswerReceived);
                    firstResultsReceived = true;

                    sendCounter(SOLOMON_AVIA_SEARCH_SUCCESS_WITH_OFFERS);
                }

                yield put(setTDAnswer(tdAnswer));
            } else if (tdAnswer) {
                yield put(updateAviaSearchProgress(tdAnswer.progress));
            }

            if (
                cont !== null &&
                tdAnswerIsEmpty &&
                !emptyTDAnswerCancellationTask
            ) {
                emptyTDAnswerCancellationTask = yield fork(
                    stopPollingByTimeout,
                );
            }

            if (!tdAnswerIsEmpty && emptyTDAnswerCancellationTask) {
                yield cancel(emptyTDAnswerCancellationTask);
                emptyTDAnswerCancellationTask = undefined;
            }
        } catch (ex) {
            if (retriesLeft-- <= 0 || cont === null) {
                yield put(stopSearch());

                if (isUnknownAxiosError(ex) && ex?.response?.status === 400) {
                    sendCounter(SOLOMON_AVIA_SEARCH_FAILED_WRONGSTART);
                } else {
                    sendCounter(SOLOMON_AVIA_SEARCH_FAILED);
                }

                return;
            }
        }

        yield delay(POLLING_DELAY);
    }
}

function isEmptyTDAnswer(data: IAviaTDAnswer): boolean {
    return !data?.variants?.fares?.length;
}

function* getCountryRestrictions(): SagaIterator {
    yield put(getCountryRestrictionsActions.reset());

    const aviaContext: IAviaContext = yield select(getAviaContext);
    const fromPointKey = aviaContext.from?.pointKey;
    const toPointKey = aviaContext.to?.pointKey;
    const fromCountryTitle = aviaContext.from?.countryTitle;
    const toCountryTitle = aviaContext.to?.countryTitle;

    if (
        !fromCountryTitle ||
        !toCountryTitle ||
        !needShowCountryRestrictions(fromCountryTitle, toCountryTitle) ||
        !fromPointKey ||
        (!isSettlementKey(fromPointKey) && !isStationKey(fromPointKey)) ||
        !toPointKey ||
        (!isSettlementKey(toPointKey) && !isStationKey(toPointKey))
    ) {
        return;
    }

    try {
        yield put(getCountryRestrictionsActions.request());

        const countryRestrictionsResponse: IGetExtendedCountryRestrictionsResponse =
            yield call(aviaBrowserProvider.getExtendedCountryRestrictions, {
                fromPointKey,
                toPointKey,
            });

        yield put(
            getCountryRestrictionsActions.success(countryRestrictionsResponse),
        );
    } catch (ex) {
        yield put(getCountryRestrictionsActions.failure(ex));
    }
}
