import {call, put, select, takeLatest} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';

import {OLD_SERVICE_HOST} from 'projects/buses/constants/url/oldService';
import {CODE_AND_STATUS_MAP} from 'projects/buses/constants/statuses/searchBooking';

import {
    EBusesSearchBookingInfoStatus,
    IBusesSearchBookingInfo,
} from 'reducers/buses/types/search/IBusesSearchBookingInfoStore';

import {StoreInterface} from 'reducers/storeTypes';
import {getSearchBookingInfoActions} from 'reducers/buses/search/bookingInfo/actions';

import requestRide from 'projects/buses/utilities/api/requestRide';
import appData from 'utilities/appData/appData';
import history from 'utilities/browserHistory/browserHistory';
import getSearchQuery from 'projects/buses/utilities/search/getSearchQuery';
import {externalUrl} from 'utilities/url';
import {restoreYtpReferer} from 'utilities/ytpReferer';
import {isUnknownAxiosError} from 'utilities/error';

function* handleRequestSearchBookingInfo({
    payload: rideId,
}: ReturnType<typeof getSearchBookingInfoActions.request>) {
    const bookingInfo: IBusesSearchBookingInfo = {
        id: rideId,
        status: EBusesSearchBookingInfoStatus.BAD_REQUEST,
    };

    if (!history) {
        return;
    }

    try {
        const {location} = history;
        const {testBookingInfoStatus} = getSearchQuery(location);

        yield call<typeof requestRide>(requestRide, {
            id: rideId,
            testStatus: testBookingInfoStatus,
        });

        bookingInfo.status = EBusesSearchBookingInfoStatus.OK;

        yield put(getSearchBookingInfoActions.success(bookingInfo));

        const {
            buses: {
                context: {value: context},
            },
        }: StoreInterface = yield select();

        if (!context) {
            return;
        }

        const ytpReferer = restoreYtpReferer();
        const bookUrl = externalUrl(
            `${
                OLD_SERVICE_HOST[
                    appData.isProductionEnv ? 'PRODUCTION' : 'TESTING'
                ]
            }ride/${context.from.key}/${context.to.key}/${
                context.when
            }/${rideId}`,
            {
                utm_source: 'travel-portal',
                utm_medium: 'bus-search',
                utm_campaign: 'button',
                ytp_referer: ytpReferer,
            },
        );

        window.location.href = bookUrl;
    } catch (err) {
        if (isUnknownAxiosError(err) && err.response?.status) {
            bookingInfo.status =
                CODE_AND_STATUS_MAP[err.response?.status] ??
                EBusesSearchBookingInfoStatus.BAD_REQUEST;
        } else {
            bookingInfo.status = EBusesSearchBookingInfoStatus.BAD_REQUEST;
        }

        yield put(getSearchBookingInfoActions.failure(bookingInfo));
    }
}

export default function* requestSearchBookingInfoSaga() {
    yield takeLatest(
        getType(getSearchBookingInfoActions.request),
        handleRequestSearchBookingInfo,
    );
}
