import {takeEvery, call, put} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {StatusCodes} from 'server/api/HotelsBookAPI/constants/statusCodes';

import {ICheckOrderAuthorizationResponse} from 'server/api/OrdersAPI/types/ICheckOrderAuthorization';

import {
    checkOrderAuthorizationActions,
    createOrderAuthorizationActions,
} from 'reducers/common/checkOrderAuthorization/actions';

import {isUnknownAxiosError} from 'utilities/error';

import {ordersBrowserProvider} from 'serviceProvider/orders/ordersBrowserProvider';

/* Check Authorization */
function* checkOrderAuthorization(
    action: ReturnType<typeof checkOrderAuthorizationActions.request>,
) {
    try {
        const response: ICheckOrderAuthorizationResponse = yield call(
            ordersBrowserProvider.checkOrderAuthorization,
            {
                id: action.payload,
            },
        );

        yield put(checkOrderAuthorizationActions.success(response));
    } catch (e) {
        if (
            isUnknownAxiosError(e) &&
            e.response &&
            e.response.status === StatusCodes.NEED_AUTHORIZATION_STATUS_CODE &&
            e.response.data
        ) {
            const orderAuthorization = e.response
                .data as ICheckOrderAuthorizationResponse;

            yield put(
                checkOrderAuthorizationActions.failure(orderAuthorization),
            );

            return;
        }

        yield put(checkOrderAuthorizationActions.failure(null));
    }
}

/* Create Authorization */
function* createOrderAuthorization(
    action: ReturnType<typeof createOrderAuthorizationActions.request>,
) {
    try {
        yield call(
            ordersBrowserProvider.createOrderAuthorization,
            action.payload,
        );

        yield put(createOrderAuthorizationActions.success());
    } catch (e) {
        yield put(createOrderAuthorizationActions.failure());
    }
}

export default function* (): SagaIterator {
    yield takeEvery(
        getType(checkOrderAuthorizationActions.request),
        checkOrderAuthorization,
    );
    yield takeEvery(
        getType(createOrderAuthorizationActions.request),
        createOrderAuthorization,
    );
}
