import {
    takeEvery,
    takeLatest,
    put,
    all,
    call,
    select,
} from 'redux-saga/effects';
import {batchActions} from 'redux-batched-actions';
import cuid from 'cuid';
import {getType} from 'typesafe-actions';
import {negate} from 'lodash';
import {SagaIterator} from 'redux-saga';

import {
    SEARCH_SUGGEST_LIMIT,
    HOTELS_SUGGEST_DEFAULT_LANGUAGE,
    HOTELS_SUGGEST_DEFAULT_DOMAIN,
} from 'constants/hotels';

import {GeoIdType} from 'types/hotels/hotel/IGeoRegion';
import {IHotelIdentifier} from 'types/hotels/hotel/IHotel';
import {ISearchSuggestResponse} from 'server/api/HotelsSearchAPI/types/ISearchSuggest';
import {TGeoRegionSlug} from 'types/hotels/geoRegion/IGeoRegionInfo';
import {EGeoLocationStatus} from 'types/common/browserGeolocation';

import {updateSearchInformation} from 'reducers/hotels/searchInformation/actions';
import {
    logSearchSuggestAction,
    fetchSearchSuggestActions,
    fetchAndSelectSearchSuggestAction,
} from 'reducers/hotels/searchSuggests/actions';
import {ISearchSuggest} from 'reducers/hotels/searchSuggests/reducer';

import getHotelsSearchSuggest from 'selectors/common/hotelsSuggestSelector';

import isNearbySuggest from 'components/SearchSuggest/utilities/isNearbySuggest';

import {hotelsSearchService} from 'serviceProvider';

const DEFAULT_REQUEST_INDEX = 1;

interface ICreateSearchSuggestRequestPayload extends IHotelIdentifier {
    query?: string;
    geoId?: GeoIdType;
    regionSlug?: TGeoRegionSlug;
    userCoordinates?: string;
    geoLocationStatus?: EGeoLocationStatus;
}

const getSearchSuggestLogInfo = function* () {
    const {
        sessionId = cuid(),
        requestIndex = DEFAULT_REQUEST_INDEX,
    }: ISearchSuggest = yield select(getHotelsSearchSuggest);

    return {
        sessionId,
        requestIndex,
    };
};

const createSearchSuggestRequest = function* ({
    query,
    geoId,
    permalink,
    hotelSlug,
    regionSlug,
    userCoordinates,
    geoLocationStatus,
}: ICreateSearchSuggestRequestPayload) {
    const {sessionId, requestIndex} = yield getSearchSuggestLogInfo();
    const {data} = yield call(hotelsSearchService.provider().searchSuggest, {
        query,
        geoId,
        regionSlug,
        permalink,
        hotelSlug,
        sessionId,
        requestIndex,
        userCoordinates,
        geoLocationStatus,
        pathname: window.location.pathname,
        limit: SEARCH_SUGGEST_LIMIT,
        language: HOTELS_SUGGEST_DEFAULT_LANGUAGE,
        domain: HOTELS_SUGGEST_DEFAULT_DOMAIN,
    });

    return data;
};

const fetchAndSelectHotelsSuggestByObjectInfo = function* (
    action: ReturnType<typeof fetchAndSelectSearchSuggestAction>,
) {
    const {
        payload: {
            geoId,
            permalink,
            hotelSlug,
            regionSlug,
            userCoordinates,
            geoLocationStatus,
            canSelectNearbyHotel,
        },
    } = action;

    if (geoId || permalink || hotelSlug || regionSlug || userCoordinates) {
        try {
            const timeMarker = Date.now();
            const searchSuggestResponse: ISearchSuggestResponse =
                yield createSearchSuggestRequest({
                    geoId,
                    permalink,
                    hotelSlug,
                    regionSlug,
                    userCoordinates,
                    geoLocationStatus,
                });

            const firstSuggestItem = searchSuggestResponse?.items.find(
                canSelectNearbyHotel
                    ? // если используем для выбора саггеста в форме, то берем первый "отели рядом" итем
                      isNearbySuggest
                    : // в случае когда заполняем саггест на странице поиска из ответа бэка не должен быть выбран элемент "отели рядом"
                      negate(isNearbySuggest),
            );

            if (firstSuggestItem) {
                const {
                    name,
                    redirectParams: {
                        hotelSlug: firstSuggestItemHotelSlug,
                        permalink: firstSuggestItemHotelPermalink,
                    },
                } = firstSuggestItem;

                yield put(
                    batchActions([
                        /* Update searchSuggest items */
                        fetchSearchSuggestActions.success({
                            items: [firstSuggestItem],
                            timeMarker,
                        }),
                        /* Update geoObject on searchInformation state */
                        updateSearchInformation({
                            geoObject: {
                                name,
                                geoId,
                                regionSlug,
                            },
                            hotel: {
                                name: name || '',
                                permalink: firstSuggestItemHotelPermalink || '',
                                hotelSlug: firstSuggestItemHotelSlug || '',
                            },
                            activeSuggest: firstSuggestItem,
                        }),
                    ]),
                );
            }
        } catch {
            yield put(fetchSearchSuggestActions.failure());
        }
    }
};

const logSearchSuggest = function* (
    action: ReturnType<typeof logSearchSuggestAction>,
) {
    try {
        const {
            payload: {selectedId, isUserInput, isManualClick, isTrustedUser},
        } = action;
        const {sessionId, requestIndex} = yield getSearchSuggestLogInfo();

        yield call(hotelsSearchService.provider().logSearchSuggest, {
            isUserInput,
            isManualClick,
            isTrustedUser,
            sessionId,
            selectedId,
            requestIndex,
        });
    } catch {}
};

const fetchHotelsSuggest = function* (
    action: ReturnType<typeof fetchSearchSuggestActions.request>,
) {
    const {
        payload: {fetchParams},
    } = action;

    try {
        const {inputValue} = fetchParams;
        const timeMarker = Date.now();
        const {items} = yield createSearchSuggestRequest({
            query: inputValue,
        });

        yield put(
            fetchSearchSuggestActions.success({
                items,
                timeMarker,
            }),
        );
    } catch {
        yield put(fetchSearchSuggestActions.failure());
    }
};

export default function* (): SagaIterator {
    yield all([
        yield takeEvery(
            getType(fetchSearchSuggestActions.request),
            fetchHotelsSuggest,
        ),
        yield takeLatest(
            getType(fetchAndSelectSearchSuggestAction),
            fetchAndSelectHotelsSuggestByObjectInfo,
        ),
        yield takeLatest(getType(logSearchSuggestAction), logSearchSuggest),
    ]);
}
