import {takeEvery, put, select} from 'redux-saga/effects';
import _uniqBy from 'lodash/uniqBy';
import _debounce from 'lodash/debounce';
import {SagaIterator} from 'redux-saga';

import {
    IPreviousFormFieldValue,
    IPreviousSearchFormSetFieldValuePayload,
} from 'reducers/common/previousSearchForm/types';

import * as actions from 'reducers/common/previousSearchForm/actions';
import {defaultFormValue} from 'reducers/common/previousSearchForm/consts';

import {previousFormsValuesSelector} from 'selectors/common/previousFormValuesSelector';

import validateSearchFormFromStorage from './utilities/validateSearchFormFromStorage';
import {
    syncStateWithStorage,
    getStateFromStorage,
} from './utilities/searchFormProvider';

const MAX_SUGGEST_ITEMS_COUNT = 4;
const SET_STORAGE_DEBOUNCE_TIME = 200;

type TPrepareFieldBeforeSetParams = Pick<
    IPreviousSearchFormSetFieldValuePayload,
    'fieldValue' | 'fieldType' | 'uniqueValueName'
> & {
    currentField: IPreviousFormFieldValue[];
};

const prepareFieldBeforeSet = ({
    currentField,
    fieldValue,
    fieldType,
    uniqueValueName,
}: TPrepareFieldBeforeSetParams): IPreviousFormFieldValue[] | undefined => {
    switch (fieldType) {
        case 'from':
        case 'to': {
            const updatedField = [fieldValue, ...currentField]
                .slice(0, MAX_SUGGEST_ITEMS_COUNT)
                .map(suggestItem => ({...suggestItem, isPreviousSearch: true}));

            return _uniqBy(updatedField, uniqueValueName);
        }
    }
};

const fillPreviousSearchForm = function* (): SagaIterator {
    const preparedForms = getStateFromStorage();
    const isValidFormsFromStorage =
        validateSearchFormFromStorage(preparedForms);

    if (isValidFormsFromStorage) {
        yield put(actions.previousSearchFormSetPreparedForms({preparedForms}));
    }
};

const syncStateWithDebounce = _debounce(
    syncStateWithStorage,
    SET_STORAGE_DEBOUNCE_TIME,
);

const setPreviousSearchFormValue = function* (
    action: ReturnType<typeof actions.previousSearchFormSetFieldValue>,
): SagaIterator {
    const {
        payload: {formName, fieldType, fieldValue, uniqueValueName},
    } = action;

    const reduxState = yield select();
    const state = previousFormsValuesSelector(reduxState);
    const currentForm = state[formName] || defaultFormValue();
    const currentField = currentForm[fieldType];
    const updatedField = prepareFieldBeforeSet({
        currentField,
        fieldValue,
        fieldType,
        uniqueValueName,
    });

    const preparedForms = {
        ...state,
        [formName]: {
            ...currentForm,
            [fieldType]: updatedField,
        },
    };

    yield put(actions.previousSearchFormSetPreparedForms({preparedForms}));

    syncStateWithDebounce(preparedForms);
};

export default function* (): SagaIterator {
    yield takeEvery(
        actions.previousSearchFormSetFieldValue,
        setPreviousSearchFormValue,
    );
    yield takeEvery(actions.previousSearchFormFillForm, fillPreviousSearchForm);
}
