import _isEmpty from 'lodash/isEmpty';
import _isObject from 'lodash/isObject';
import _isArray from 'lodash/isArray';
import _isUndefined from 'lodash/isUndefined';

import {
    AVIA_SEARCH_FORM_NAME,
    BUSES_SEARCH_FORM_NAME,
    HOTELS_SEARCH_FORM_NAME,
    RAILWAYS_SEARCH_FORM_NAME,
} from 'constants/reduxForm';

import {
    IPreviousFormFieldValue,
    IPreviousSearchFormStore,
} from 'reducers/common/previousSearchForm/types';

type TSupportedSearchForms = {
    formName: keyof IPreviousSearchFormStore;
    requiredDirectionProps: (keyof IPreviousFormFieldValue)[];
    fields: ('from' | 'to')[];
}[];

/* Validate helpers */

const validateDirectionField = (
    field: IPreviousFormFieldValue[],
    requiredDirectionProps: (keyof IPreviousFormFieldValue)[],
): boolean =>
    _isArray(field) &&
    field.every(
        fieldValue =>
            _isObject(fieldValue) &&
            requiredDirectionProps.every(
                directionValue =>
                    _isUndefined(fieldValue[directionValue]) === false,
            ),
    );

/* Validate Configs */

const supportedSearchForms: TSupportedSearchForms = [
    {
        formName: AVIA_SEARCH_FORM_NAME,
        requiredDirectionProps: ['type', 'title', 'countryTitle', 'pointKey'],
        fields: ['from', 'to'],
    },
    {
        formName: RAILWAYS_SEARCH_FORM_NAME,
        requiredDirectionProps: ['objId', 'objType', 'title'],
        fields: ['from', 'to'],
    },
    {
        formName: HOTELS_SEARCH_FORM_NAME,
        requiredDirectionProps: ['objId', 'name', 'description'],
        fields: ['to'],
    },
    {
        formName: BUSES_SEARCH_FORM_NAME,
        requiredDirectionProps: ['objId', 'objType', 'title'],
        fields: ['from', 'to'],
    },
];

export default (searchForms: IPreviousSearchFormStore = {}): boolean => {
    if (_isEmpty(searchForms) && _isObject(searchForms) === false) {
        return false;
    }

    return supportedSearchForms.every(
        ({formName, requiredDirectionProps, fields}) => {
            const currentForm = searchForms[formName];

            if (!currentForm || _isEmpty(currentForm)) {
                return false;
            }

            return fields.every(fieldName =>
                validateDirectionField(
                    currentForm[fieldName],
                    requiredDirectionProps,
                ),
            );
        },
    );
};
