import {call, select, put} from '@redux-saga/core/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    fetchUserUnreadMailCount,
    setUserUnreadMailCount,
} from 'reducers/common/userUnreadMailCount/actions';

import watchActions from 'sagas/trains/helpers/watchActions';

import {logError} from 'utilities/logger/logError';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {browserProvider as mailApiBrowserProvider} from 'serviceProvider/mailApi/browserProvider';

function* handleFetchUserUnreadMailCount(): SagaIterator {
    try {
        const isAuth = yield select(state => state.common.userInfo.isAuth);

        if (isAuth) {
            const data = yield call(mailApiBrowserProvider.counters);

            yield put(setUserUnreadMailCount(data.counters.unread));
        }
    } catch (e) {
        const error = unknownToErrorOrUndefined(e);

        if (error && error.message !== 'Network Error') {
            // Too many messages with this error, could not reproduce the problem
            logError(
                {
                    message:
                        '[YATRAVEL] Ошибка запроса за количеством непрочитанных писем',
                    block: 'fetchUserUnreadMailCount',
                },
                error,
            );
        }
    }
}

export default function* userUnreadMailCount() {
    yield watchActions(handleFetchUserUnreadMailCount, [
        getType(fetchUserUnreadMailCount),
    ]);
}
