import {takeEvery, all, call, put, select} from 'redux-saga/effects';
import {SagaIterator} from 'redux-saga';

import IGetFavoriteHotelsRequestParams from 'server/api/HotelsFavoritesAPI/types/IGetFavoriteHotelsRequestParams';
import IAddFavoriteHotelRequestParams from 'server/api/HotelsFavoritesAPI/types/IAddFavoriteHotelRequestParams';
import {TRemoveFavoriteHotelsRequestParams} from 'server/api/HotelsFavoritesAPI/types/IRemoveFavoriteHotelsRequestParams';
import IGetSharedFavoriteHotelsRequestParams from 'server/api/HotelsFavoritesAPI/types/IGetSharedFavoriteHotelsRequestParams';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {TPromiseReturnType} from 'types/utilities';
import {TAddFavoriteHotelResponse} from 'server/api/HotelsFavoritesAPI/types/TAddFavoriteHotelResponse';

import {HOTELS_FAVORITES_ACTION_TYPES} from 'reducers/favorites/actionTypes';
import {
    addFavoriteHotelAction,
    removeFavoriteHotelAction,
    getFavoritesInfoActions,
    pollingHotelsOffersActions,
    changeFavoriteHotelsOffsetAction,
    changeHotelsIsFavoriteActions,
    removeFavoriteHotelsAction,
    getSharedFavoriteHotelsActions,
} from 'reducers/favorites/actions';
import {IFavoritesInfoState} from 'reducers/favorites/reducer';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {getAttributionParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';

import {hotelsFavoritesService} from 'serviceProvider';

const getFavoriteHotelRequest = function* (
    params: IGetFavoriteHotelsRequestParams,
): SagaIterator {
    const data = yield call(
        hotelsFavoritesService.provider().getFavoriteHotels,
        params,
    );

    return data;
};

const getFavoriteHotels = function* (
    action: ReturnType<typeof getFavoritesInfoActions.request>,
): SagaIterator {
    const {offerSearchParams}: IFavoritesInfoState = yield select(
        getFavoritesInfoSelector,
    );
    const {withInfiniteScroll, ...request} = action.payload;

    const {data} = yield call(getFavoriteHotelRequest, {
        ...request,
        ...offerSearchParams,
    });

    yield put(
        getFavoritesInfoActions.success({
            data,
            withInfiniteScroll,
        }),
    );
    yield put(changeFavoriteHotelsOffsetAction(action.payload.offset));

    if (!data.offerSearchProgress.finished) {
        const permalinks = data.hotels.map(
            (hotel: IHotelWithOffers) => hotel.hotel.permalink,
        );

        yield put(pollingHotelsOffersActions.request(permalinks));
    }
};

const getSharedFavoriteHotelsRequest = function* (
    params: IGetSharedFavoriteHotelsRequestParams,
): SagaIterator {
    try {
        const data = yield call(
            hotelsFavoritesService.provider().getSharedFavoriteHotels,
            params,
        );

        return data;
    } catch (e) {
        yield put(getSharedFavoriteHotelsActions.failure());
    }
};

const getSharedFavoriteHotels = function* (
    action: ReturnType<typeof getSharedFavoriteHotelsActions.request>,
): SagaIterator {
    const {offerSearchParams}: IFavoritesInfoState = yield select(
        getFavoritesInfoSelector,
    );

    const result = yield call(getSharedFavoriteHotelsRequest, {
        ...action.payload,
        ...offerSearchParams,
    });

    yield put(getSharedFavoriteHotelsActions.success(result.data));
    yield put(changeFavoriteHotelsOffsetAction(action.payload.offset));

    const permalinks = result.data.hotels.map(
        (hotel: IHotelWithOffers) => hotel.hotel.permalink,
    );

    yield put(pollingHotelsOffersActions.request(permalinks));
};

const addFavoriteHotelRequest = function* (
    params: IAddFavoriteHotelRequestParams,
): SagaIterator<TAddFavoriteHotelResponse> {
    const requestParams = {
        ...params,
        ...getAttributionParams(),
    };

    const data = yield call(
        hotelsFavoritesService.provider().addFavoriteHotel,
        requestParams,
    );

    return data;
};

const addFavoriteHotel = function* (
    action: ReturnType<typeof addFavoriteHotelAction>,
): SagaIterator {
    const result: TAddFavoriteHotelResponse = yield call(
        addFavoriteHotelRequest,
        {
            permalink: action.payload.permalink,
        },
    );

    if ('error' in result && result.error) return;

    yield put(changeHotelsIsFavoriteActions.success(action.payload.permalink));
};

const removeFavoriteHotelRequest = function* (
    params: TRemoveFavoriteHotelsRequestParams,
): SagaIterator {
    const requestParams = {
        ...params,
        ...getAttributionParams(),
    };

    const {removeFavoriteHotels} = hotelsFavoritesService.provider();

    const data: TPromiseReturnType<typeof removeFavoriteHotels> = yield call(
        removeFavoriteHotels,
        requestParams,
    );

    return data;
};

const removeFavoriteHotel = function* (
    action: ReturnType<typeof removeFavoriteHotelAction>,
): SagaIterator {
    yield call(removeFavoriteHotelRequest, {
        permalink: action.payload.permalink,
    });

    yield put(changeHotelsIsFavoriteActions.success(action.payload.permalink));
};

const removeFavoriteHotels = function* (
    action: ReturnType<typeof removeFavoriteHotelsAction>,
): SagaIterator {
    yield call(removeFavoriteHotelRequest, {
        categoryId: action.payload.categoryId,
    });
};

export default function* (): SagaIterator {
    yield all([
        yield takeEvery(
            HOTELS_FAVORITES_ACTION_TYPES.START_GET_FAVORITES_INFO_REQUEST,
            getFavoriteHotels,
        ),
        yield takeEvery(
            HOTELS_FAVORITES_ACTION_TYPES.ADD_FAVORITE_HOTEL,
            addFavoriteHotel,
        ),
        yield takeEvery(
            HOTELS_FAVORITES_ACTION_TYPES.REMOVE_FAVORITE_HOTEL,
            removeFavoriteHotel,
        ),
        yield takeEvery(
            HOTELS_FAVORITES_ACTION_TYPES.REMOVE_FAVORITE_HOTELS,
            removeFavoriteHotels,
        ),
        yield takeEvery(
            HOTELS_FAVORITES_ACTION_TYPES.START_GET_SHARED_FAVORITES_HOTELS_REQUEST,
            getSharedFavoriteHotels,
        ),
    ]);
}
