import {
    call,
    delay,
    put,
    race,
    select,
    take,
    takeLatest,
} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import IGetFavoriteHotelsOffersResponse from 'server/api/HotelsFavoritesAPI/types/IGetFavoriteHotelsOffersResponse';
import IGetFavoriteHotelsOffersRequestParams from 'server/api/HotelsFavoritesAPI/types/IGetFavoriteHotelsOffersRequestParams';

import {
    pollingHotelsOffersActions,
    stopPollingHotelsOffersAction,
} from 'reducers/favorites/actions';
import {IFavoritesInfoState} from 'reducers/favorites/reducer';

import {getFavoritesInfoSelector} from 'selectors/favorites/getFavoritesInfo';

import {getAttributionParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';
import sendMetrikaExtraVisitAndUserParams from 'projects/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';

import {hotelsFavoritesService} from 'serviceProvider';

/* Constants */
const MIN_POLLING_DELAY = 200;
const POLLING_DELAY = 2000;

const getHotelsOffersPollingRequestParams = function* () {
    const {context, offerSearchParams}: IFavoritesInfoState = yield select(
        getFavoritesInfoSelector,
    );

    return {
        context,
        ...offerSearchParams,
    };
};

const startHotelsOffersPolling = function* (
    action: ReturnType<typeof pollingHotelsOffersActions.request>,
): SagaIterator {
    const {payload} = action;
    const debugAndMetaAttributionParamsByLocation = getAttributionParams();
    const startHotelsOffersPollingRequest =
        hotelsFavoritesService.provider().getFavoriteHotelsOffers;

    while (true) {
        try {
            const searchParams = yield call(
                getHotelsOffersPollingRequestParams,
            );
            const hotelsOffersPollingRequestParams: IGetFavoriteHotelsOffersRequestParams =
                {
                    ...debugAndMetaAttributionParamsByLocation,
                    ...searchParams,
                    permalinks: payload,
                };

            const {
                data: pollingResponse,
            }: {data: IGetFavoriteHotelsOffersResponse} = yield call(
                startHotelsOffersPollingRequest,
                hotelsOffersPollingRequestParams,
            );

            yield put(pollingHotelsOffersActions.success(pollingResponse));

            if (pollingResponse.offerSearchProgress?.finished) {
                if (pollingResponse.extraVisitAndUserParams) {
                    sendMetrikaExtraVisitAndUserParams(
                        pollingResponse.extraVisitAndUserParams,
                    );
                }

                yield put(stopPollingHotelsOffersAction());
            } else {
                yield delay(
                    Math.max(
                        pollingResponse.nextPollingRequestDelayMs ||
                            POLLING_DELAY,
                        MIN_POLLING_DELAY,
                    ),
                );
            }
        } catch (e) {
            console.log(e);
            yield put(pollingHotelsOffersActions.failure());
            yield put(stopPollingHotelsOffersAction());
        }
    }
};

const watchHotelsOffersPolling = function* (
    action: ReturnType<typeof pollingHotelsOffersActions.request>,
) {
    yield race({
        startAction: call(startHotelsOffersPolling, action),
        stopAction: take(getType(stopPollingHotelsOffersAction)),
    });
};

export default function* (): SagaIterator {
    yield takeLatest(
        getType(pollingHotelsOffersActions.request),
        watchHotelsOffersPolling,
    );
}
