import {takeEvery, call, put, delay} from 'redux-saga/effects';
import {AxiosResponse} from 'axios';
import get from 'lodash/get';
import {SagaIterator} from 'redux-saga';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IOrderRefundableInfo} from 'server/api/HotelsBookAPI/types/IOrderRefundableInfo';

/* Actions Types */
import {
    CANCEL_ORDER,
    FETCH_REFUNDABLE_INFO,
} from 'reducers/hotels/bookAndPayPage/orderCancelAndRefundableInfo/actionTypes';
/* Actions */
import {
    startCancelOrderActions,
    fetchRefundableInfoActions,
} from 'reducers/hotels/bookAndPayPage/orderCancelAndRefundableInfo/actions';

/* Utilities */
import {reachGoal} from 'utilities/metrika';
import {getCSRFTokenHeaderGenerator} from 'utilities/csrfToken/getCSRFTokenHeaderGenerator';
/* Api Client */
import APIClientProvider from '../utilities/cancelOrderAPIClient';

const apiClientInstance = new APIClientProvider();

/* Constants */
const DELAY_AFTER_SUCCESS_CANCEL_ORDER = 2000;

/* Fetch refundable info */

function* fetchRefundableInfo(
    action: ReturnType<typeof fetchRefundableInfoActions.request>,
) {
    try {
        const response: AxiosResponse = yield call(
            apiClientInstance.fetchOrderRefundableInfo,
            action.payload,
        );
        const payload: IOrderRefundableInfo = get(response, 'data.data');

        yield put(fetchRefundableInfoActions.success(payload));
    } catch (e) {
        yield put(fetchRefundableInfoActions.failure());
    }
}

/* Cancel Order */

function* cancelOrder(
    action: ReturnType<typeof startCancelOrderActions.request>,
): SagaIterator {
    try {
        const csrfTokenHeader = yield call(getCSRFTokenHeaderGenerator);

        yield call(
            apiClientInstance.cancelOrder,
            action.payload,
            csrfTokenHeader,
        );

        reachGoal(EHotelsGoal.ORDER_CANCEL_SUCCESS);

        yield delay(DELAY_AFTER_SUCCESS_CANCEL_ORDER);
        yield put(startCancelOrderActions.success());
    } catch (e) {
        reachGoal(EHotelsGoal.ORDER_CANCEL_ERROR);

        yield put(startCancelOrderActions.failure());
    }
}

export default function* (): SagaIterator {
    yield takeEvery(FETCH_REFUNDABLE_INFO, fetchRefundableInfo);
    yield takeEvery(CANCEL_ORDER, cancelOrder);
}
