import {batchActions} from 'redux-batched-actions';
import {call, put, race, select, take, takeEvery} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {EAppActions} from 'constants/platforms/TPlatforms';
import {EOfferStatus} from 'projects/hotels/constants/EOfferStatus';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {
    ICreateOrderRequest,
    IOrderError,
    TCreateOrderErrorResponse,
    TCreateOrderResponse,
} from 'types/hotels/book/ICreateOrder';
import {
    EPromoCodesApplicationType,
    IAppliedPromoCodesInfo,
} from 'types/hotels/book/IApplyPromoCodes';

import {fetchOfferInfoByTokenAction} from 'reducers/hotels/bookAndPayPage/offerInfoByToken/actions';
import {
    createOrderActions,
    getOrderInfoActions,
    resetOrderInfoAction,
    stopCreateOrderActions,
} from 'reducers/hotels/bookAndPayPage/orderInfo/actions';
import {IBookPromoCodesReducer} from 'reducers/hotels/bookAndPayPage/promoCodesInfo/reducer';
import {resetBookUserSelectAction} from 'reducers/hotels/bookAndPayPage/bookUserSelectOptions/actions';
import {IPlatformState} from 'reducers/avia/platform/reducer';

import {getBookPromoCodes} from 'selectors/hotels/book/getBookPromoCodes';
import {aviaPlatformSelector} from 'selectors/avia/platform/aviaPlatformSelector';

import {reachGoal} from 'utilities/metrika';
import {getCSRFTokenHeaderGenerator} from 'utilities/csrfToken/getCSRFTokenHeaderGenerator';
import {createNormalizedGuestsFormCopy} from '../utilities/createNormalizedGuestsFormCopy';
import {getPaymentTestContextToken} from 'utilities/url/getPaymentTestContextToken';

import {createCoordinator} from 'contexts/PlatformContext';

import {hotelBookService} from 'serviceProvider';

const prepareErrorResponse = (
    errorResponse?: TCreateOrderErrorResponse,
): IOrderError => {
    return {
        code:
            errorResponse?.data?.error?.code ??
            EOfferStatus.UNKNOWN_OFFER_ERROR,
        offerInfo: errorResponse?.data?.error?.offerInfo,
    };
};

function preparePromoCodes(
    promoCodes?: IAppliedPromoCodesInfo,
): string[] | undefined {
    return promoCodes?.codeApplicationResults
        .filter(
            promoCode => promoCode.type === EPromoCodesApplicationType.SUCCESS,
        )
        .map(promoCode => promoCode.code);
}

const createHotelOrder = function* (
    action: ReturnType<typeof createOrderActions.request>,
): SagaIterator {
    const {payload: actionPayload} = action;
    const {deviceType, finishPaymentPagePath, ...payload} = actionPayload;

    payload.formValues = createNormalizedGuestsFormCopy(payload.formValues);

    try {
        const {data}: IBookPromoCodesReducer = yield select(getBookPromoCodes);

        const requestPayload: ICreateOrderRequest = {
            ...payload,
            promoCodes: preparePromoCodes(data),
            paymentTestContextToken: getPaymentTestContextToken(),
        };

        /* Create order request */
        const csrfTokenHeader = yield call(getCSRFTokenHeaderGenerator);
        const {data: orderRequestResponse}: TCreateOrderResponse = yield call(
            hotelBookService.provider().createOrder,
            requestPayload,
            csrfTokenHeader,
        );

        const {orderId} = orderRequestResponse;

        const platform: IPlatformState = yield select(aviaPlatformSelector);

        reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_SUCCESS);

        if (platform.isTravelApp) {
            const coordinator = createCoordinator(platform);

            coordinator.doAction(EAppActions.CREATE_HOTEL_ORDER, orderId);

            /**
             * Полностью отдаем управление Travel App
             * Сбрасываем состояние страницы жо исходных
             * https://st.yandex-team.ru/TRAVELFRONT-7956
             */
            yield put(resetOrderInfoAction());

            return;
        }

        /* Start polling order action */
        yield put(
            getOrderInfoActions.request({
                orderId,
                finishPaymentPagePath,
            }),
        );

        /* Reset user payment info form */
        yield put(resetBookUserSelectAction());
    } catch (e) {
        const error = prepareErrorResponse(e.response);
        const {code} = error;

        if (code === EOfferStatus.UNKNOWN_OFFER_ERROR) {
            reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_ERROR, {
                hotels: {orderError: code.toLowerCase()},
            });

            /* Create order finished with error */
            yield put(createOrderActions.failure());
        } else {
            reachGoal(EHotelsGoal.BOOK_CREATE_ORDER_ERROR, {
                hotels: {orderError: code && code.toLowerCase()},
            });

            yield put(
                batchActions([
                    /* Fetch offer info finished with error */
                    fetchOfferInfoByTokenAction.failure({error}),
                    /* Reset order */
                    resetOrderInfoAction(),
                ]),
            );
        }
    }
};

function* createHotelOrderOrStop(
    action: ReturnType<typeof createOrderActions.request>,
): SagaIterator {
    yield race([
        call(createHotelOrder, action),
        take(getType(stopCreateOrderActions)),
    ]);
}

export default function* (): SagaIterator {
    /* Create order watcher */
    yield takeEvery(
        getType(createOrderActions.request),
        createHotelOrderOrStop,
    );
}
