import {
    takeEvery,
    call,
    put,
    select,
    race,
    take,
    delay,
} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    IStartPaymentParams,
    EStartPaymentCustomerSource,
} from 'server/api/HotelsBookAPI/types/TStartPayment';

import {StoreInterface} from 'reducers/storeTypes';
import {
    startOrderPaymentActions,
    getOrderInfoActions,
    stopOrderPaymentActions,
} from 'reducers/hotels/bookAndPayPage/orderInfo/actions';

import getDeviceType from 'selectors/common/deviceTypeSelector';

import {getCSRFTokenHeaderGenerator} from 'utilities/csrfToken/getCSRFTokenHeaderGenerator';
import {getStartPaymentTestContextToken} from 'utilities/url/getStartPaymentTestContextToken';

import {hotelBookService} from 'serviceProvider';

const startOrderPayment = function* (
    action: ReturnType<typeof startOrderPaymentActions.request>,
): SagaIterator {
    const {payload} = action;
    const reduxState: StoreInterface = yield select();
    const {isMobile} = getDeviceType(reduxState);

    try {
        const {orderId, finishPaymentPagePath} = payload;

        /* Create start payment order request */
        const csrfTokenHeader = yield call(getCSRFTokenHeaderGenerator);
        const data: IStartPaymentParams = {
            orderId,
            returnUrl: finishPaymentPagePath || '',
            customerSource: isMobile
                ? EStartPaymentCustomerSource.MOBILE
                : EStartPaymentCustomerSource.DESKTOP,
            paymentTestContextToken: getStartPaymentTestContextToken(),
        };

        yield call(
            hotelBookService.provider().startPayment,
            data,
            csrfTokenHeader,
        );

        /**
         * Запускаем поллинг с небольшой задержкой,
         * чтобы заказ всегда успевал перейти в статус IN_PROGRESS
         */
        yield delay(2000);

        /* Start polling order after payment invoice */
        yield put(
            getOrderInfoActions.request({orderId, isPaymentStarted: true}),
        );
    } catch {
        /* Created payment invoice with error */
        yield put(startOrderPaymentActions.failure());
    }
};

function* startOrderPaymentOrStop(
    action: ReturnType<typeof startOrderPaymentActions.request>,
): SagaIterator {
    yield race([
        call(startOrderPayment, action),
        take(getType(stopOrderPaymentActions)),
    ]);
}

export default function* (): SagaIterator {
    /* Start payment order watcher */
    yield takeEvery(
        getType(startOrderPaymentActions.request),
        startOrderPaymentOrStop,
    );
}
