import {
    all,
    call,
    take,
    race,
    put,
    delay,
    select,
    takeEvery,
} from 'redux-saga/effects';
import {SagaIterator} from 'redux-saga';
import {getType} from 'typesafe-actions';
import moment from 'moment';
import {isEqual} from 'lodash';

import {DEFAULT_ADULTS_COUNT} from 'constants/hotels';

import {ECalendarEmptyPriceReason} from 'types/common/calendarPrice/ICalendarPrice';
import {
    ICalendarPricesRequest,
    ICalendarPricesResponse,
} from 'server/api/HotelSearchAPI/types/ICalendarPrices';

import {
    getHotelCalendarPricesActions,
    startHotelCalendarPricesPoolingAction,
    stopHotelCalendarPricesAction,
} from 'reducers/hotels/hotelPage/calendarPrices/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {getHotelInfo} from 'selectors/hotels/hotel/mainTab/getHotelInfo';
import {getHotelCalendarPrices} from 'selectors/hotels/hotel/getHotelCalendarPrices';
import searchFormSelector from 'selectors/hotels/searchForm/searchFormSelector';
import experimentsSelector from 'selectors/common/experimentsSelector';

import {ROBOT} from 'utilities/dateUtils/formats';

import {hotelSearchService} from 'serviceProvider';

const CALENDAR_PRICES_POLLING_DELAY = 1000;
const CALENDAR_PRICES_INTERVAL_IN_DAYS = 90;

const startCalendarPricesPolling = function* (
    request: ICalendarPricesRequest,
): SagaIterator {
    yield put(getHotelCalendarPricesActions.request(request));

    let context;

    while (true) {
        try {
            const response: ICalendarPricesResponse = yield call(
                hotelSearchService.provider().getCalendarPrices,
                {...request, context},
            );

            const {prices, finished} = response;

            context = response.context;

            const {data: calendarPrices} = yield select(getHotelCalendarPrices);
            const newPrices = prices.reduce((result, priceInfo) => {
                result[priceInfo.date] = priceInfo.price || {
                    emptyPriceReason: ECalendarEmptyPriceReason.NO_OFFERS,
                };

                return result;
            }, calendarPrices || {});

            yield put(
                getHotelCalendarPricesActions.success({
                    ...newPrices,
                }),
            );

            if (finished) {
                yield put(stopHotelCalendarPricesAction());
            } else {
                yield delay(CALENDAR_PRICES_POLLING_DELAY);
            }
        } catch (e) {
            yield put(getHotelCalendarPricesActions.failure());
            yield put(stopHotelCalendarPricesAction());
        }
    }
};

const watchCalendarPricesPolling = function* () {
    const state: StoreInterface = yield select();
    const {data: hotelInfo} = getHotelInfo(state);
    const searchForm = searchFormSelector(state);
    const calendarPrices = getHotelCalendarPrices(state);
    const {hotelsCalendarPrices} = experimentsSelector(state);

    if (!hotelsCalendarPrices) {
        return;
    }

    if (!hotelInfo) {
        return;
    }

    const {
        searchParams,
        hotel: {hotelSlug},
    } = hotelInfo;

    const adults =
        searchForm.adultsField || searchParams?.adults || DEFAULT_ADULTS_COUNT;
    const childrenAges =
        searchForm.childrenAgesField || searchParams?.childrenAges;
    const checkinDate = searchForm.startDateField || searchParams?.checkinDate;

    if (!checkinDate || !hotelSlug) {
        return;
    }

    const checkinMoment = moment(checkinDate, ROBOT);

    let startMoment = checkinMoment.subtract(1, 'month');

    if (startMoment.isBefore(moment())) {
        startMoment = moment();
    }

    const endMoment = moment(startMoment).add(
        CALENDAR_PRICES_INTERVAL_IN_DAYS,
        'days',
    );

    const request = {
        hotelSlug,
        startDate: startMoment.format(ROBOT),
        endDate: endMoment.format(ROBOT),
        adults,
        childrenAges,
    };

    if (isEqual(calendarPrices.request, request)) {
        return;
    }

    yield put(stopHotelCalendarPricesAction());
    yield race({
        startAction: call(startCalendarPricesPolling, request),
        stopAction: take(getType(stopHotelCalendarPricesAction)),
    });
};

export default function* (): SagaIterator {
    yield all([
        yield takeEvery(
            getType(startHotelCalendarPricesPoolingAction),
            watchCalendarPricesPolling,
        ),
    ]);
}
