import {
    takeLatest,
    put,
    all,
    call,
    delay,
    select,
    CallEffect,
    race,
    take,
} from 'redux-saga/effects';
import {SagaIterator} from 'redux-saga';
import {getType} from 'typesafe-actions';

import {
    IHotelInfo,
    IHotelInfoRequestParams,
} from 'server/api/HotelSearchAPI/types/IHotelInfo';

import {HOTEL_INFO_ACTION_TYPES} from 'reducers/hotels/hotelPage/hotelInfo/actionTypes';
import {IHotelInfoReducer} from 'reducers/hotels/hotelPage/hotelInfo/reducer';
import {
    getHotelInfoActions,
    getHotelInfoOffersActions,
    getSimilarHotelsActions,
    getAdditionalHotelInfoAction,
    syncHotelInfoSearchParamsWithLocation,
    stopHotelInfoActionsAction,
} from 'reducers/hotels/hotelPage/hotelInfo/actions';
import {resetHotelImagesAction} from 'reducers/hotels/hotelPage/hotelImages/actions';
import {
    getHotelReviewsActions,
    resetHotelReviewsAction,
} from 'reducers/hotels/hotelPage/reviews/list/actions';
import {StoreInterface} from 'reducers/storeTypes';
import {startHotelCalendarPricesPoolingAction} from 'reducers/hotels/hotelPage/calendarPrices/actions';

import {getHotelInfo} from 'selectors/hotels/hotel/mainTab/getHotelInfo';

import {isUnknownAxiosError} from 'utilities/error';
import sendMetrikaExtraVisitAndUserParams from 'projects/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';

import {hotelSearchService} from 'serviceProvider';

/* Constants */

const DELAY_AFTER_FETCH_HOTEL_INFO = 200;

export const fetchHotelAdditionHotelInfoIfNeed = function* () {
    const reduxState: StoreInterface = yield select();
    const {data} = getHotelInfo(reduxState);
    const canStartHotelInfoOffersPolling =
        !data?.offersInfo?.offerSearchProgress?.finished;
    const canStartSimilarHotelsInfoPolling =
        !data?.similarHotelsInfo?.offerSearchProgress.finished;

    if (canStartHotelInfoOffersPolling) {
        yield put(getHotelInfoOffersActions.request());
    }

    if (canStartSimilarHotelsInfoPolling) {
        yield put(getSimilarHotelsActions.request());
    }

    yield put(startHotelCalendarPricesPoolingAction());
};

/* AdditionHotelInfo */
const watchStartFetchAdditionHotelInfo = function* () {
    /* Offers */
    yield fetchHotelAdditionHotelInfoIfNeed();

    /* Location */
    yield put(syncHotelInfoSearchParamsWithLocation());
};

/* HotelInfo */
const startHotelInfoRequest = function* (
    requestParams: IHotelInfoRequestParams,
): Generator<
    CallEffect<IHotelInfo>,
    IHotelInfo | undefined,
    IHotelInfoReducer
> {
    const {data} = yield call(
        hotelSearchService.provider().getHotelInfo,
        requestParams,
    );

    return data;
};

const startFetchHotelInfo = function* (
    payload: IHotelInfoRequestParams,
): SagaIterator {
    try {
        yield put(resetHotelImagesAction());

        const hotelInfo = yield call(startHotelInfoRequest, payload);

        sendMetrikaExtraVisitAndUserParams(hotelInfo.extraVisitAndUserParams);

        yield put(getHotelInfoActions.success(hotelInfo));
        yield put(resetHotelReviewsAction());
        yield put(getHotelReviewsActions.success(hotelInfo.reviewsInfo));

        yield delay(DELAY_AFTER_FETCH_HOTEL_INFO);
        yield put(getAdditionalHotelInfoAction());
    } catch (error) {
        const status = isUnknownAxiosError(error)
            ? error.response?.status
            : undefined;

        yield put(getHotelInfoActions.failure(status));
    }
};

const watchStartFetchHotelInfo = function* (
    action: ReturnType<typeof getHotelInfoActions.request>,
) {
    const {payload} = action;

    yield race({
        startAction: call(startFetchHotelInfo, payload),
        stopAction: take(getType(stopHotelInfoActionsAction)),
    });
};

/* Total sagas */
export default function* (): SagaIterator {
    yield all([
        yield takeLatest(
            HOTEL_INFO_ACTION_TYPES.START_HOTEL_INFO_REQUEST,
            watchStartFetchHotelInfo,
        ),
        yield takeLatest(
            HOTEL_INFO_ACTION_TYPES.GET_ADDITIONAL_HOTEL_INFO,
            watchStartFetchAdditionHotelInfo,
        ),
    ]);
}
