import {
    call,
    delay,
    put,
    race,
    select,
    take,
    takeLatest,
} from 'redux-saga/effects';
import _getFp from 'lodash/fp/get';
import {isEqual} from 'lodash';
import {getType, PayloadAction} from 'typesafe-actions';
import {batchActions} from 'redux-batched-actions';
import {SagaIterator} from 'redux-saga';

import {
    PAGE_HOTEL_COUNT,
    PRICED_HOTEL_LIMIT,
    TOTAL_HOTEL_LIMIT,
} from 'projects/hotels/constants/searchPage';

import {ISearchHotelsInfo} from 'reducers/hotels/searchPage/search/types';
import {EStartSearchReason} from 'types/hotels/search/ISearchReason';
import {EGeoLocationStatus} from 'types/common/browserGeolocation';
import {ISearchPageTopHotelQueryParams} from 'types/hotels/common/IQueryParams';
import {EActiveHotelSource} from 'types/hotels/hotel/IActiveHotel';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';

import {StoreInterface} from 'reducers/storeTypes';
import {
    clearHotels,
    clearNotFinishedHotels,
    clearSearchBeforeNewSearch,
    searchHotelsActions,
    stopSearchHotels,
    syncSearchParamsWithLocation,
} from 'reducers/hotels/searchPage/search/actions';
import {
    fillFilters,
    fillResetActions,
    resetFiltersToInitial,
    updateCountFiltersActions,
} from 'reducers/hotels/searchPage/filters/actions';
import {
    fillHotelList,
    resetHotelList,
    resetHotelListToInitial,
} from 'reducers/hotels/searchPage/hotelList/actions';
import {
    setActiveHotel,
    setDefaultActiveHotel,
} from 'reducers/hotels/searchPage/selection/actions';
import {
    clearMapHotels,
    fillMap,
    resetMapToInitial,
} from 'reducers/hotels/searchPage/map/actions';
import {
    fillSortInfo,
    resetSortToInitial,
} from 'reducers/hotels/searchPage/sort/actions';

import {
    getSearchInfo,
    GetSearchInfoType,
} from 'selectors/hotels/search/searchHotels/getSearchInfo';
import {getMap, GetMapType} from 'selectors/hotels/search/map/getMap';
import {
    getHotelList,
    GetHotelListType,
} from 'selectors/hotels/search/hotelList/getHotelList';
import {
    GetFiltersType,
    getHotelsFilters,
} from 'selectors/hotels/search/filters/hotelsFiltersSelector';
import {getSortInfo} from 'selectors/hotels/search/sort/getSortInfo';
import {getActiveHotelPermalink} from 'selectors/hotels/search/selection/getActiveHotelPermalink';
import experimentsSelector from 'selectors/common/experimentsSelector';
import deviceTypeSelector from 'selectors/common/deviceTypeSelector';

import {getOfferRequestParams} from 'sagas/hotels/utilities/getSearchPageOfferRequestParams';

import {convertBoundsToString} from 'components/YandexMaps/utilities';
import {getSearchParamsByLocation} from 'projects/hotels/utilities/getSearchPageParamsByLocation/getSearchPageParamsByLocation';
import {getAttributionParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';
import sendMetrikaExtraVisitAndUserParams from 'projects/hotels/utilities/metrika/sendMetrikaExtraVisitAndUserParams';
import sendMetrikaSearchRequestsTimes from 'projects/hotels/utilities/metrika/sendMetrikaSearchRequestsTimes';
import {getFilterParams} from 'projects/hotels/utilities/getSearchPageFilterParams/getSearchPageFilterParams';

import {hotelsSearchService} from 'serviceProvider';

/* Constants */
const MIN_SEARCH_POLLING_DELAY = 200;
const SEARCH_POLLING_DELAY = 2000;
const FIRST_POLL_EPOCH = 0;
const FIRST_POLL_ITERATION = 0;
const DEFAULT_POLL_CONTEXT = undefined;

/* Helpers */

interface IHotelLimits {
    pageHotelCount: number;
    pricedHotelLimit: number;
    totalHotelLimit: number;
}

const hotelLimits: IHotelLimits = {
    pageHotelCount: PAGE_HOTEL_COUNT,
    pricedHotelLimit: PRICED_HOTEL_LIMIT,
    totalHotelLimit: TOTAL_HOTEL_LIMIT,
};

const checkSearchFinish = _getFp<
    ISearchHotelsInfo,
    'offerSearchProgress',
    'finished'
>(['offerSearchProgress', 'finished']);

/* Polling Params */
const checkFirstIterationByFirstEpoch = (
    currentPollEpoch: number,
    search: ReturnType<GetSearchInfoType>,
) => currentPollEpoch === FIRST_POLL_EPOCH && search.isLoading;

const getPollIteration = (
    currentPollEpoch: number,
    search: ReturnType<GetSearchInfoType>,
    isFirstIterationByFirstEpoch: boolean,
): number => {
    const {data: searchData} = search;

    if (isFirstIterationByFirstEpoch) {
        return FIRST_POLL_ITERATION;
    }

    if (searchData) {
        const {pollIteration, pollEpoch} = searchData;

        if (pollEpoch === currentPollEpoch) {
            return pollIteration + 1;
        }
    }

    return FIRST_POLL_ITERATION;
};

const getPollContext = (
    search: ReturnType<GetSearchInfoType>,
    isFirstIterationByFirstEpoch: boolean,
): string | undefined => {
    const {data: searchData} = search;

    if (isFirstIterationByFirstEpoch) {
        return DEFAULT_POLL_CONTEXT;
    }

    return searchData ? searchData.context : DEFAULT_POLL_CONTEXT;
};

const getPollingParams = (reduxState: StoreInterface, pollEpoch: number) => {
    const search: ReturnType<GetSearchInfoType> = getSearchInfo(reduxState);
    const isFirstIterationByFirstEpoch = checkFirstIterationByFirstEpoch(
        pollEpoch,
        search,
    );
    const pollIteration = getPollIteration(
        pollEpoch,
        search,
        isFirstIterationByFirstEpoch,
    );
    const context = getPollContext(search, isFirstIterationByFirstEpoch);

    return {
        pollIteration,
        context,
        pollEpoch,
    };
};

/* Filters Params */
export const getFilterParamsByState = (reduxState: StoreInterface) => {
    const {permanentFilters}: ReturnType<GetFiltersType> =
        getHotelsFilters(reduxState);

    return getFilterParams(permanentFilters);
};

const getTopHotel = (
    reduxState: StoreInterface,
): ISearchPageTopHotelQueryParams => {
    const {data} = getSearchInfo(reduxState);

    return data?.topHotelSlug ? {topHotelSlug: data.topHotelSlug} : {};
};

/* Navigation Token */
const getNavigationToken = (
    reduxState: StoreInterface,
    startSearchReason: EStartSearchReason,
) => {
    const {
        navigationToken: currentNavigationToken,
    }: ReturnType<GetHotelListType> = getHotelList(reduxState);

    if (
        startSearchReason === EStartSearchReason.NAVIGATION_TOKEN ||
        startSearchReason === EStartSearchReason.MAP_BOUNDS
    ) {
        const navigationToken = currentNavigationToken
            ? currentNavigationToken
            : undefined;

        return {
            navigationToken,
        };
    }

    return {};
};

/* Map */
const getMapBounds = (
    reduxState: StoreInterface,
    startSearchReason: EStartSearchReason,
    pollEpoch: number,
) => {
    const search: ReturnType<GetSearchInfoType> = getSearchInfo(reduxState);
    const {mapBounds}: ReturnType<GetMapType> = getMap(reduxState);
    const isFirstIterationByFirstEpoch = checkFirstIterationByFirstEpoch(
        pollEpoch,
        search,
    );

    /* Clear bbox for first epoch */
    if (
        isFirstIterationByFirstEpoch &&
        startSearchReason !== EStartSearchReason.QUERY_WITH_SAME_GEO
    ) {
        return {
            bbox: undefined,
        };
    }

    const bbox = mapBounds ? convertBoundsToString(mapBounds) : undefined;

    return {
        bbox,
    };
};

/* Sort */
const getSortParamsByState = (reduxState: StoreInterface) => {
    const {sortInfo, sortOrigin} = getSortInfo(reduxState);
    const selectedSortId = sortInfo?.selectedSortId;
    const sortInfoFromServer = sortInfo?.sortOrigin;

    return {
        selectedSortId,
        sortOrigin: sortInfoFromServer || sortOrigin,
        geoLocationStatus:
            sortInfoFromServer || sortOrigin
                ? EGeoLocationStatus.AVAILABLE
                : EGeoLocationStatus.UNKNOWN,
    };
};

/* Generator Helpers */

const getPollEpoch = function* (startSearchReason: EStartSearchReason) {
    const {data: searchData}: ReturnType<GetSearchInfoType> = yield select(
        getSearchInfo,
    );

    if (
        !searchData ||
        startSearchReason === EStartSearchReason.MOUNT ||
        startSearchReason === EStartSearchReason.QUERY_BY_LOCATION ||
        startSearchReason === EStartSearchReason.QUERY_WITH_SAME_GEO
    ) {
        return FIRST_POLL_EPOCH;
    }

    return searchData.pollEpoch + 1;
};

const getSearchParams = function* (
    pollEpoch: number,
    startSearchReason: EStartSearchReason,
) {
    const reduxState: StoreInterface = yield select();
    const pollingParams = getPollingParams(reduxState, pollEpoch);
    const isFirstPollIteration = pollingParams.pollIteration === 0;
    const canUseSearchParamsFromLocation =
        startSearchReason === EStartSearchReason.MOUNT ||
        startSearchReason === EStartSearchReason.QUERY_BY_LOCATION;

    if (canUseSearchParamsFromLocation && isFirstPollIteration) {
        /* Start search with params from location */
        return {
            ...pollingParams,
            ...getSearchParamsByLocation(),
        };
    }

    if (
        startSearchReason === EStartSearchReason.QUERY_WITH_SAME_GEO &&
        isFirstPollIteration
    ) {
        return {
            ...pollingParams,
            ...getSearchParamsByLocation(),
            ...getSortParamsByState(reduxState),
            ...getMapBounds(reduxState, startSearchReason, pollEpoch),
        };
    }

    /* Start search with params from redux state */
    return {
        ...pollingParams,
        ...getOfferRequestParams(reduxState),
        ...getFilterParamsByState(reduxState),
        ...getNavigationToken(reduxState, startSearchReason),
        ...getMapBounds(reduxState, startSearchReason, pollEpoch),
        ...getSortParamsByState(reduxState),
        ...getTopHotel(reduxState),
    };
};

const getMetrikaSearchRequestsTimes = function* () {
    const reduxState: StoreInterface = yield select();
    const startSearchTime = getSearchInfo(reduxState).startSearchTime;

    const endSearchTime = Date.now();
    const requestsTimeSumFrontend = startSearchTime
        ? endSearchTime - startSearchTime
        : null;

    return requestsTimeSumFrontend;
};

/* Update states before search polling */
const clearAndResetSearchResults = function* (
    startSearchReason: EStartSearchReason,
) {
    const {isMobile} = yield select(deviceTypeSelector);

    switch (startSearchReason) {
        case EStartSearchReason.MOUNT:
        case EStartSearchReason.QUERY_BY_LOCATION: {
            yield put(
                batchActions([
                    resetFiltersToInitial(),
                    resetHotelListToInitial(),
                    resetMapToInitial(),
                    resetSortToInitial(),
                    clearSearchBeforeNewSearch(),
                    setActiveHotel(),
                    setDefaultActiveHotel(),
                ]),
            );

            break;
        }

        case EStartSearchReason.QUERY_WITH_SAME_GEO: {
            yield put(
                batchActions([
                    resetHotelListToInitial(),
                    clearMapHotels(),
                    clearHotels(),
                    setActiveHotel(),
                    setDefaultActiveHotel(),
                    clearNotFinishedHotels(),
                ]),
            );

            break;
        }

        case EStartSearchReason.FILTERS: {
            yield put(
                batchActions([
                    resetHotelListToInitial(),
                    clearMapHotels(),
                    clearHotels(),
                    setActiveHotel(),
                    setDefaultActiveHotel(),
                ]),
            );

            break;
        }

        case EStartSearchReason.NAVIGATION_TOKEN: {
            if (isMobile) {
                yield put(batchActions([clearNotFinishedHotels()]));
            } else {
                yield put(
                    batchActions([resetHotelList(), clearNotFinishedHotels()]),
                );
            }

            break;
        }
        case EStartSearchReason.MAP_BOUNDS:
        case EStartSearchReason.SORT: {
            yield put(
                batchActions([resetHotelList(), clearNotFinishedHotels()]),
            );

            break;
        }
    }
};

function* setDefaultHotelSelection(hotels: IHotelWithOffers[]) {
    const experiments: ReturnType<typeof experimentsSelector> = yield select(
        experimentsSelector,
    );

    if (!experiments.hotelsTopSnippet) {
        return;
    }

    const searchedByUser = hotels.find(hotel => hotel.searchedByUser);

    if (searchedByUser) {
        yield put(
            setDefaultActiveHotel({
                permalink: searchedByUser.hotel.permalink,
                source: EActiveHotelSource.EXTERNAL,
            }),
        );

        const activeHotel = getActiveHotelPermalink(yield select());

        if (!activeHotel) {
            yield put(
                setActiveHotel({
                    permalink: searchedByUser.hotel.permalink,
                    source: EActiveHotelSource.EXTERNAL,
                }),
            );
        }
    }
}

const startSearchPolling = function* (
    action: ReturnType<typeof searchHotelsActions.request>,
): SagaIterator {
    const {payload} = action;
    const {startSearchReason} = payload;
    const {pageHotelCount} = hotelLimits;

    const {isMobile}: ReturnType<typeof deviceTypeSelector> = yield select(
        deviceTypeSelector,
    );
    const debugAndMetaAttributionParamsByLocation = getAttributionParams();
    const startSearchRequest = hotelsSearchService.provider().searchHotels;
    const pollEpoch = yield call(getPollEpoch, startSearchReason);
    let requestsTimeSumBackend = 0;

    yield call(clearAndResetSearchResults, startSearchReason);

    while (true) {
        try {
            const {
                permanentFilters,
                resetFilterWasUsed,
            }: ReturnType<typeof getHotelsFilters> = yield select(
                getHotelsFilters,
            );
            const searchParams = yield call(
                getSearchParams,
                pollEpoch,
                startSearchReason,
            );
            const searchHotelsPollingParams = {
                startSearchReason,
                ...hotelLimits,
                ...debugAndMetaAttributionParamsByLocation,
                ...searchParams,
            };

            const {data: pollingResponse}: {data: ISearchHotelsInfo} =
                yield call(startSearchRequest, searchHotelsPollingParams);
            const isFinishedSearch = checkSearchFinish(pollingResponse);

            const {currentFilters}: ReturnType<typeof getHotelsFilters> =
                yield select(getHotelsFilters);
            const filtersWereChanged = !isEqual(
                getFilterParams(currentFilters),
                getFilterParams(permanentFilters),
            );

            const actions: PayloadAction<any, any>[] = [
                searchHotelsActions.success(pollingResponse),
                fillHotelList({
                    pageHotelCount,
                    pollingResponse,
                    withInfiniteScroll: isMobile,
                }),
                fillMap(pollingResponse),
                fillSortInfo(pollingResponse),
                fillResetActions(pollingResponse.filterInfo.resetFilterInfo),
            ];

            if (filtersWereChanged) {
                yield put(updateCountFiltersActions.request());
            } else {
                actions.push(fillFilters(pollingResponse));
            }

            yield call(setDefaultHotelSelection, pollingResponse.hotels);

            requestsTimeSumBackend +=
                pollingResponse.timingInfo.currentRequestDurationMs;

            /* With batch */
            yield put(batchActions(actions));

            /* Without batch */
            const preventScroll = Boolean(
                (resetFilterWasUsed && !pollingResponse?.hotels?.length) ||
                    isMobile,
            );

            yield put(
                syncSearchParamsWithLocation({
                    preventScroll,
                }),
            );

            if (isFinishedSearch) {
                const reduxState: StoreInterface = yield select();
                const {passToMetricHotelsSearchPageSearchtime} =
                    experimentsSelector(reduxState);

                if (passToMetricHotelsSearchPageSearchtime) {
                    const requestsTimeSumFrontend = yield call(
                        getMetrikaSearchRequestsTimes,
                    );

                    sendMetrikaSearchRequestsTimes({
                        requestsTimeSumBackend,
                        requestsTimeSumFrontend,
                    });
                }

                sendMetrikaExtraVisitAndUserParams(
                    pollingResponse.extraVisitAndUserParams,
                );

                yield put(stopSearchHotels());
            } else {
                yield delay(
                    Math.max(
                        pollingResponse.nextPollingRequestDelayMs ||
                            SEARCH_POLLING_DELAY,
                        MIN_SEARCH_POLLING_DELAY,
                    ),
                );
            }
        } catch (e) {
            yield put(searchHotelsActions.failure());
            yield put(stopSearchHotels());
        }
    }
};

const watchSearchPolling = function* (
    action: ReturnType<typeof searchHotelsActions.request>,
) {
    yield race({
        startAction: call(startSearchPolling, action),
        stopAction: take(getType(stopSearchHotels)),
    });
};

export default function* (): SagaIterator {
    yield takeLatest(getType(searchHotelsActions.request), watchSearchPolling);
}
