import {delay, select, call, takeLatest} from 'redux-saga/effects';
import {ParsedQuery} from 'query-string';
import {omitBy, isUndefined} from 'lodash';
import {SagaIterator} from 'redux-saga';

import {ISearchPageQueryParams} from 'types/hotels/common/IQueryParams';
import {ISearchInformation as ISearchInformationReducer} from 'reducers/hotels/searchInformation/types';

import {SEARCH_HOTELS_ACTION_TYPES} from 'reducers/hotels/searchPage/search/actionTypes';
import {syncSearchParamsWithLocation} from 'reducers/hotels/searchPage/search/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {
    getSearchInfo,
    GetSearchInfoType,
} from 'selectors/hotels/search/searchHotels/getSearchInfo';
import {
    getHotelsFilters,
    GetFiltersType,
} from 'selectors/hotels/search/filters/hotelsFiltersSelector';
import {getMap, GetMapType} from 'selectors/hotels/search/map/getMap';
import {
    getHotelList,
    GetHotelListType,
} from 'selectors/hotels/search/hotelList/getHotelList';
import {
    getSortInfo,
    TGetSortInfo,
} from 'selectors/hotels/search/sort/getSortInfo';
import {getSearchInformation} from 'selectors/hotels/search/searchInformation/searchInformationSelector';
import deviceTypeSelector from 'selectors/common/deviceTypeSelector';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {getSearchPageQueryByInfo} from 'projects/hotels/utilities/getSearchPageQueryByInfo/getSearchPageQueryByInfo';
import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';
import history from 'utilities/browserHistory/browserHistory';
import {getAttributionQueryParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';
import {getLastSearchTimeMarkerQuery} from 'projects/hotels/utilities/getLastSearchTimeMarkerQuery/getLastSearchTimeMarkerQuery';
import {getSearchPageLayoutViewQueryParams} from 'projects/hotels/utilities/getSearchPageLayoutViewParams/getSearchPageLayoutViewParams';
import {getStoredSrcQueryParams} from 'utilities/srcParams/srcParams';

/* Types */
type FiltersType = ReturnType<GetFiltersType>;
type SearchInfoType = ReturnType<GetSearchInfoType>;
type MapType = ReturnType<GetMapType>;
type HotelListType = ReturnType<GetHotelListType>;
type TSortInfoReducer = ReturnType<TGetSortInfo>;

/* Constants */
const SYNC_SEARCH_PARAMS_DELAY = 500;

/* Helpers */
export const checkDifferenceBetweenQueryAndSearchInfoParams = (
    queryByLocation: ParsedQuery,
    queryParamsBySearchInfo: ISearchPageQueryParams,
): boolean => {
    const queryByLocationKeys = Object.keys(queryByLocation);
    const queryParamsBySearchInfoKeys = Object.keys(
        queryParamsBySearchInfo,
    ) as (keyof ISearchPageQueryParams)[];

    if (queryByLocationKeys.length !== queryParamsBySearchInfoKeys.length) {
        return true;
    }

    return !queryParamsBySearchInfoKeys.every(paramKey => {
        const queryByLocationValue = queryByLocation[paramKey];
        const queryBySearchInfoValue = String(
            queryParamsBySearchInfo[paramKey],
        );

        return queryByLocationValue === queryBySearchInfoValue;
    });
};

export const startSyncSearchParamsWithLocation = function* ({
    payload,
}: ReturnType<typeof syncSearchParamsWithLocation>) {
    /* Delay after search request */
    yield delay(SYNC_SEARCH_PARAMS_DELAY);

    /* Get current search and filters params */
    const reduxState: StoreInterface = yield select();
    const searchInfo: SearchInfoType = getSearchInfo(reduxState);
    const filtersInfo: FiltersType = getHotelsFilters(reduxState);
    const mapInfo: MapType = getMap(reduxState);
    const hotelList: HotelListType = getHotelList(reduxState);
    const sortInfoReducer: TSortInfoReducer = getSortInfo(reduxState);
    const searchInformationReducer: ISearchInformationReducer =
        getSearchInformation(reduxState);
    const {isMobile} = deviceTypeSelector(reduxState);

    /* Sync params with location */
    if (history) {
        const sortOrigin =
            searchInformationReducer?.activeSuggest?.redirectParams
                ?.sortOrigin || sortInfoReducer?.sortInfo?.sortOrigin;
        const selectedSortId = sortInfoReducer?.sortInfo?.selectedSortId;
        const queryParamsBySearchInfo: ISearchPageQueryParams = omitBy(
            {
                ...getStoredSrcQueryParams(),
                ...getAttributionQueryParams(),
                ...getLastSearchTimeMarkerQuery(),
                ...getSearchPageLayoutViewQueryParams(),
                ...getSearchPageQueryByInfo({
                    filters: filtersInfo.permanentFilters,
                    offerSearchParams: searchInfo.data?.offerSearchParams,
                    geoId: searchInfo.data?.searchRegion?.geoId,
                    mapBounds: mapInfo.mapBoundsBySearch,
                    navigationToken: isMobile
                        ? undefined
                        : hotelList.navigationToken,
                    selectedSortId: selectedSortId,
                    sortOrigin,
                    topHotelSlug: searchInfo.data?.topHotelSlug,
                    searchPagePollingId: searchInfo.data?.searchPagePollingId,
                }),
            },
            isUndefined,
        );
        const queryByLocation = getQueryByLocation(history.location);
        const hasDifferenceBetweenQueryAndSearchInfoParams =
            checkDifferenceBetweenQueryAndSearchInfoParams(
                queryByLocation,
                queryParamsBySearchInfo,
            );

        if (hasDifferenceBetweenQueryAndSearchInfoParams) {
            const preventScroll = payload?.preventScroll;

            yield call(
                updateLocationWithQuery,
                queryParamsBySearchInfo,
                history.location,
                {
                    useOnlyNextQuery: true,
                    replace: true,
                    needSaveLocationState: true,
                    scrollTop: preventScroll ? null : 0,
                },
            );
        }
    }
};

export default function* (): SagaIterator {
    yield takeLatest(
        SEARCH_HOTELS_ACTION_TYPES.SYNC_SEARCH_PARAMS_WITH_LOCATION,
        startSyncSearchParamsWithLocation,
    );
}
