import {all, call, put, takeLatest} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import moment from 'moment-timezone';

import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {
    IBusesApiContext,
    isFilledContext,
} from 'types/buses/common/IBusesContext';

import {
    requestPreviousSearchesAction,
    requestPreviousSearchesItemFailureAction,
    requestPreviousSearchesItemSuccessAction,
} from 'reducers/buses/previousSearches/actions';

import priceComparator from 'selectors/buses/search/utilities/sort/priceComparator';
import departureComparator from 'selectors/buses/search/utilities/sort/departureComparator';

import {ROBOT} from 'utilities/dateUtils/formats';
import requestSearchSegments from 'projects/buses/utilities/api/requestSearchSegments';
import requestContext from 'projects/buses/utilities/api/requestContext';
import {getNow} from 'utilities/dateUtils';
import getWhenMoment from 'utilities/dateUtils/when/getWhenMoment';

function* handleRequestPreviousSearches({
    payload,
}: ReturnType<typeof requestPreviousSearchesAction>) {
    yield all(
        payload.map(({from, to, when}, index) =>
            call(function* () {
                try {
                    const apiContext: IBusesApiContext = yield call<
                        typeof requestContext
                    >(requestContext, {
                        params: {
                            fromSlug: from.slug,
                            toSlug: to.slug,
                        },
                    });

                    if (!apiContext.from.timezone) {
                        return;
                    }

                    const departureNowMoment = moment(getNow()).tz(
                        apiContext.from.timezone,
                    );

                    const today = departureNowMoment.format(ROBOT);

                    const searchWhen = when || today;

                    const context = {
                        ...apiContext,
                        when: searchWhen,
                        originWhen: searchWhen,
                    };

                    if (!isFilledContext(context)) {
                        return;
                    }

                    const previousSearchWhenMoment = getWhenMoment(
                        searchWhen,
                        context.from.timezone,
                    );
                    const isToday = previousSearchWhenMoment.isSame(
                        departureNowMoment,
                        'day',
                    );

                    context.when = searchWhen;
                    context.originWhen = searchWhen;

                    const segments: IBusesSegment[] = yield call<
                        typeof requestSearchSegments
                    >(requestSearchSegments, {context});

                    yield put(
                        requestPreviousSearchesItemSuccessAction({
                            index,
                            price: segments.length
                                ? segments.sort(priceComparator)[0].price.amount
                                : null,
                            times: segments
                                .sort(departureComparator)
                                .map(({departureTime}) =>
                                    Number(moment(departureTime)),
                                ),
                            timezone: context.from.timezone,
                            when: searchWhen,
                            isToday,
                        }),
                    );
                } catch (err) {
                    yield put(requestPreviousSearchesItemFailureAction(index));
                }
            }),
        ),
    );
}

export default function* requestPreviousSearchesSaga() {
    yield takeLatest(
        getType(requestPreviousSearchesAction),
        handleRequestPreviousSearches,
    );
}
