import {BATCH} from 'redux-batched-actions';
import {takeEvery, fork, cancel} from 'redux-saga/effects';
import {SagaIterator, Task} from 'redux-saga';
import {Action, PayloadAction, TypeConstant} from 'typesafe-actions';

/**
 * Функция запускает generator, если пришел экшен с типом из actionsList.
 * Поддерживает проверку экшенов из batchActions.
 */

// Не смог победить здесь TS+Redux-Saga, Parameters<typeof fork>[0] возвращает
// параметры для последнего оверлода fork

// eslint-disable-next-line @typescript-eslint/no-explicit-any
type TGenerator = any;

function runIfActionInList(
    generator: TGenerator,
    actionsList: TypeConstant[],
    needAllBatchHandle?: boolean,
): (action: PayloadAction<TypeConstant, Action[]>) => SagaIterator {
    let task: Task | undefined;

    return function* (
        action: PayloadAction<TypeConstant, Action[]>,
    ): SagaIterator {
        const {payload} = action;
        const isBatchedAction = action.type === BATCH;
        const match = isBatchedAction
            ? payload.some(({type}) => actionsList.includes(type))
            : actionsList.includes(action.type);

        if (match) {
            if (task) {
                yield cancel(task);
            }

            if (isBatchedAction) {
                if (needAllBatchHandle) {
                    task = yield fork(
                        generator,
                        payload.filter(({type}) => actionsList.includes(type)),
                    );
                } else {
                    task = yield fork(
                        generator,
                        payload.find(({type}) => actionsList.includes(type)),
                    );
                }
            } else {
                task = yield fork(generator, action);
            }
        }
    };
}

/**
 * Вспомогательная функция для саги, которая следит за списком экшенов actionsList
 * и запускает generator при диспатче экшена из списка.
 * Работает вместе с batchActions из redux-batch-actions.
 */
export default function* watchActions(
    generator: Parameters<typeof runIfActionInList>[0],
    actionsList: TypeConstant[],
    needAllBatchHandle?: boolean,
): SagaIterator {
    yield takeEvery(
        [BATCH, ...actionsList],
        runIfActionInList(generator, actionsList, needAllBatchHandle),
    );
}
