import {
    call,
    delay,
    put,
    race,
    retry,
    select,
    take,
    takeLatest,
    CallEffect,
    ForkEffect,
    RaceEffect,
    TakeEffect,
} from 'redux-saga/effects';
import {getType, ActionType} from 'typesafe-actions';
import {batchActions} from 'redux-batched-actions';
import {Action} from 'redux';
import {SagaIterator} from 'redux-saga';

import {ESearchErrorTypes} from 'projects/trains/lib/search/constants';

import {
    ITrainsSearchInfoResponse,
    ITrainsSearchRequestParams,
} from 'server/api/TrainsSearchApi/types/ITrainsSearch';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {
    isFilledTrainsSearchContext,
    ITrainsFilledSearchContext,
} from 'reducers/trains/context/types';
import {ETrainSearchMock} from 'projects/testControlPanel/pages/TrainsPage/types';

import {StoreInterface} from 'reducers/storeTypes';
import {
    stopTrainsSearch,
    trainsSearchActions,
} from 'reducers/trains/genericSearch/search/actions';
import {setTrainsContext} from 'reducers/trains/context/actions';
import {
    fillTrainsSearchFilters,
    TFillTrainsSearchFiltersActionType,
} from 'reducers/trains/genericSearch/filters/actions';

import experimentsSelector from 'selectors/common/experimentsSelector';
import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';

import scrollTo from 'utilities/dom/scrollTo';
import updateContext from 'projects/trains/lib/context/updateContext';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import {checkYandexMollyVisit} from 'utilities/checkYandexMollyVisit/checkYandexMollyVisit';
import {getTrainsFiltersByVariants} from 'projects/trains/lib/genericSearch/filters/getTrainsFiltersByVariants';
import {getMockImSearchPath} from 'utilities/testUtils/getMockImSearchPath';

import {trainsSearchProvider} from 'serviceProvider';

/* Constants */
const POLLING_DELAY = 1000;
const POLLING_RETRIES = 3;
const MAX_ATTEMPT_NUMBER_FOR_CALCULATE_DELAY = 4;

/* Helpers */
function* getSearchPollingParams(
    customContext: ITrainsFilledSearchContext | undefined,
): SagaIterator<ITrainsSearchRequestParams | undefined> {
    const reduxState: StoreInterface = yield select();
    const queryParams = getQueryByBrowserHistory();
    const {from, to, when, returnWhen, forwardSegmentId} =
        customContext || trainsContextSelector(reduxState);
    const {trainsOnlyDirect} = experimentsSelector(reduxState);

    if (from?.key && to?.key && when) {
        return {
            when,
            pointFrom: from.key,
            pointTo: to.key,
            returnWhen: forwardSegmentId && returnWhen ? returnWhen : undefined,
            pinForwardSegmentId: forwardSegmentId ?? undefined,
            isBot: checkYandexMollyVisit(queryParams),
            onlyDirect: Boolean(returnWhen) || trainsOnlyDirect,
            onlyOwnedPrices: Boolean(returnWhen),
            mockImAuto: getMockImSearchPath() === ETrainSearchMock.AUTO,
        };
    }
}

function* getUpdateSearchContextIfNeedActions(
    trainsSearchInfoResponse: ITrainsSearchInfoResponse,
    customContext: ITrainsFilledSearchContext | undefined,
): SagaIterator<Action[] | void> {
    if (!trainsSearchInfoResponse.context?.isChanged) {
        return;
    }

    const reduxState: StoreInterface = yield select();
    const searchContext = customContext || trainsContextSelector(reduxState);

    if (isFilledTrainsSearchContext(searchContext)) {
        return [
            setTrainsContext(
                updateContext(searchContext, trainsSearchInfoResponse.context),
            ),
        ];
    }
}

function* getFillSearchFiltersAction(
    trainsSearchInfoResponse: ITrainsSearchInfoResponse,
): SagaIterator<TFillTrainsSearchFiltersActionType> {
    const reduxState: StoreInterface = yield select();
    const context = trainsContextSelector(reduxState);
    const filters = getTrainsFiltersByVariants({
        context,
        status: trainsSearchInfoResponse.status,
        variants: trainsSearchInfoResponse.variants,
    });

    return fillTrainsSearchFilters(filters);
}

function* startSearchPolling(
    customContext: ITrainsFilledSearchContext | undefined,
) {
    let pollingAttemptNumber = 1;

    while (true) {
        try {
            const currentPollingDelay =
                Math.min(
                    MAX_ATTEMPT_NUMBER_FOR_CALCULATE_DELAY,
                    pollingAttemptNumber,
                ) * POLLING_DELAY;

            const searchParams: ITrainsSearchRequestParams = yield call(
                getSearchPollingParams,
                customContext,
            );
            const trainsSearchInfoResponse: ITrainsSearchInfoResponse =
                yield retry(
                    POLLING_RETRIES,
                    currentPollingDelay,
                    trainsSearchProvider.provider().search,
                    searchParams,
                );

            const actions: Action[] = [
                trainsSearchActions.success(trainsSearchInfoResponse),
            ];
            const fillSearchFiltersAction: Action = yield call(
                getFillSearchFiltersAction,
                trainsSearchInfoResponse,
            );
            const updateActions: Action[] =
                yield getUpdateSearchContextIfNeedActions(
                    trainsSearchInfoResponse,
                    customContext,
                );

            actions.push(fillSearchFiltersAction);

            if (updateActions) {
                actions.push(...updateActions);
            }

            yield put(batchActions(actions));

            if (trainsSearchInfoResponse.status === EQueryingStatus.QUERYING) {
                pollingAttemptNumber++;
                yield delay(currentPollingDelay);
            } else {
                yield put(stopTrainsSearch());
            }
        } catch (e) {
            yield put(
                trainsSearchActions.failure({
                    errorCode: ESearchErrorTypes.COMMON,
                }),
            );
            yield put(stopTrainsSearch());
        }
    }
}

function* searchPollingWorker(
    action: ActionType<typeof trainsSearchActions.request>,
): Generator<RaceEffect<TakeEffect | CallEffect>, void, void> {
    scrollTo({top: 0, left: 0});

    yield race({
        startAction: call(startSearchPolling, action.payload),
        stopAction: take(getType(stopTrainsSearch)),
    });
}

export default function* watchGenericSearch(): Generator<
    ForkEffect,
    void,
    void
> {
    yield takeLatest(getType(trainsSearchActions.request), searchPollingWorker);
}
