import {getType} from 'typesafe-actions';
import {call, delay, select, takeLatest} from 'redux-saga/effects';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';

import {
    setTrainsSearchFilterValue,
    resetAllTrainsSearchFilterValues,
} from 'reducers/trains/genericSearch/filters/actions';
import {StoreInterface} from 'reducers/storeTypes';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {getTrainsOriginalSearchFilters} from 'selectors/trains/genericSearch/filters/getTrainsOriginalSearchFilters';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';

import history from 'utilities/browserHistory/browserHistory';
import getFiltersQuery from 'projects/trains/lib/filters/getFiltersQuery';
import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';

const DELAY_BEFORE_UPDATE_LOCATION = 500;
const DELAY_BEFORE_UPDATE_LOCATION_FOR_QUERYING_STATUS = 100;

function* handleTrainsSearchFiltersChange() {
    const reduxStateBeforeDelay: StoreInterface = yield select();
    const {status} = getTrainsOriginalSearchInfo(reduxStateBeforeDelay);

    yield delay(
        status === EQueryingStatus.QUERYING
            ? DELAY_BEFORE_UPDATE_LOCATION_FOR_QUERYING_STATUS
            : DELAY_BEFORE_UPDATE_LOCATION,
    );

    const reduxStateAfterDelay: StoreInterface = yield select();
    const context = trainsContextSelector(reduxStateAfterDelay);
    const filters = getTrainsOriginalSearchFilters(reduxStateAfterDelay);

    if (!isFilledTrainsSearchContext(context) || !history) {
        return;
    }

    yield call(() => {
        if (!history) {
            return;
        }

        updateLocationWithQuery(getFiltersQuery(filters), history.location, {
            replace: true,
            needSaveLocationState: true,
            scrollTop: 0,
        });
    });
}

export default function* watchGenericSearchFiltersChange() {
    yield takeLatest(
        getType(setTrainsSearchFilterValue),
        handleTrainsSearchFiltersChange,
    );
    yield takeLatest(
        getType(resetAllTrainsSearchFilterValues),
        handleTrainsSearchFiltersChange,
    );
}
