import {cancel, delay, fork, put, select, take} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {Task} from 'redux-saga';

import {MINUTE} from 'utilities/dateUtils/constants';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import {
    cancelOrder,
    cancelOrderPolling,
    startOrder,
} from 'reducers/trains/order/actions/flow';
import {setTrainsPartner} from 'reducers/trains/common/actions';
import {setOrderError} from 'reducers/trains/order/actions/view';
import {updatePurchaseAvailability} from 'reducers/trains/actions/updatePurchaseAvailability';
import {setTrainDetails} from 'reducers/trains/order/actions/trains';

import {trainsPurchaseIsAvailableSelector} from 'selectors/trains/trainsPurchaseIsAvailableSelector';
import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';
import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import {isOrderStepAfterReserve} from 'projects/trains/lib/order/orderStatus';

function* pollingOrderAvailability() {
    try {
        let purchaseIsAvailable = trainsPurchaseIsAvailableSelector(
            yield select(),
        );

        while (purchaseIsAvailable) {
            yield delay(MINUTE);
            // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
            yield put(updatePurchaseAvailability());
            yield take(getType(setTrainsPartner));

            purchaseIsAvailable = trainsPurchaseIsAvailableSelector(
                yield select(),
            );

            if (!purchaseIsAvailable) {
                const orderStep: ReturnType<typeof orderStepSelector> =
                    yield select(orderStepSelector);

                if (isOrderStepAfterReserve(orderStep)) {
                    yield put(cancelOrderPolling());
                    yield put(setOrderError({type: EOrderErrorType.COMMON}));
                } else {
                    const currentSegmentDirectionAndIndex: ReturnType<
                        typeof currentSegmentDirectionAndIndexSelector
                    > = yield select(currentSegmentDirectionAndIndexSelector);

                    yield put(
                        setTrainDetails({
                            ...currentSegmentDirectionAndIndex,
                            data: {
                                errors: true,
                                trainDetails: null,
                                partnerError: null,
                            },
                        }),
                    );
                }
            }
        }
    } catch (e) {}
}

function* managePolling() {
    const pollingTask: Task = yield fork(pollingOrderAvailability);

    yield take(getType(cancelOrder));
    yield cancel(pollingTask);
}

export default function* watchOrderAvailability() {
    yield watchActions(managePolling, [getType(startOrder)]);
}
