import {
    all,
    call,
    cancel,
    delay,
    fork,
    put,
    select,
    take,
} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import isEmpty from 'lodash/isEmpty';
import {Task} from 'redux-saga';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';
import {E_COMMERCE_TRAINS_PURCHASE_GOAL_ID} from 'projects/trains/constants/metrika';

import {ECommerceActionType} from 'utilities/metrika/types/ecommerce';
import {TrainsInsuranceStatus} from 'server/api/TrainsBookingApi/types/TrainsInsuranceStatus';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import ECancellationReason from 'server/api/GenericOrderApi/types/common/ECancellationReason';
import {IOrderError} from 'reducers/trains/order/types';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ETrainOrderErrorCode} from 'projects/trains/components/TrainsOrderApp/components/BookError/types';
import {isTrainApiOrderError} from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/TTrainOrderError/TTrainApiOrderError';

import {StoreInterface} from 'reducers/storeTypes';
import {
    setOrderInfo,
    updateOrderInfo,
} from 'reducers/trains/order/actions/data';
import {setBookError, setOrderError} from 'reducers/trains/order/actions/view';
import {createOrderAuthorizationActions} from 'reducers/common/checkOrderAuthorization/actions';
import {
    cancelOrder,
    cancelOrderPolling,
    resumeOrderPolling,
    setPaymentCalled,
} from 'reducers/trains/order/actions/flow';
import {resetPayment} from 'reducers/trains/order/thunk/resetPayment';
import {setTimeoutError} from 'reducers/trains/order/thunk/setTimeoutError';
import {setOrderCancelledAfterPaidError} from 'reducers/trains/order/thunk/setOrderCancelledAfterPaidError';
import {setCancelOrderError} from 'reducers/trains/order/thunk/setCancelOrderError';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import orderInfoSelector from 'selectors/trains/order/orderInfoSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import {eCommercePush} from 'utilities/metrika/ecommerce';
import browserHistory from 'utilities/browserHistory/browserHistory';
import getECommerceRevenue from 'projects/trains/lib/metrika/getECommerceRevenue';
import getECommerceProductFromContext from 'projects/trains/lib/metrika/getECommerceProductFromContext';
import {getPaymentError} from 'projects/trains/lib/order/payment';
import {cancelOrder as cancelOrderApi} from 'projects/trains/lib/api/cancelOrder';
import {trainsURLs} from 'projects/trains/lib/urls';
import getFirstForwardTrainDetails from 'projects/trains/lib/complexOrder/getFirstForwardTrainDetails';
import {logError} from 'utilities/logger/logError';
import {getTrainServices} from 'projects/trains/lib/order/getTrainServices';
import {getFirstTrainService} from 'projects/trains/lib/complexOrder/getFirstTrainService';
import createLatestOrderInfoFetcher from 'projects/trains/lib/api/createLatestOrderInfoFetcher';
import normalizeGenericOrderState from 'projects/trains/lib/api/utilities/normalizeGenericOrderState';
import isFinalGenericOrderState from 'projects/trains/lib/api/utilities/isFinalGenericOrderState';
import {isUnknownAxiosError, unknownToErrorOrUndefined} from 'utilities/error';

const POLLING_INTERVAL = 3000;
const POLLING_STEPS = [ORDER_STEP.CONFIRM, ORDER_STEP.PAYMENT];
const BEFORE_PAYMENT_STATUSES = [
    EGenericOrderState.RESERVED,
    EGenericOrderState.WAITING_PAYMENT,
];

const MAX_ORDER_AUTH_CHECKS = 3;
const ORDER_AUTH_CHECK_DELAY = 1000;

function* dispatchOrderInfo(newOrderInfo: IGenericOrderInfo) {
    const {
        trains: {
            order,
            order: {orderInfo},
        },
    }: StoreInterface = yield select((state: StoreInterface) => state);

    const trainDetails = getFirstForwardTrainDetails(order);

    if (!orderInfo || !trainDetails) {
        return;
    }

    const error = getTrainServices(newOrderInfo).find(
        service => service.trainInfo.error,
    )?.trainInfo.error;

    if (error) {
        yield put(
            setBookError(
                isTrainApiOrderError(error)
                    ? error
                    : {
                          code: ETrainOrderErrorCode.AFTER_RESERVATION,
                      },
            ),
        );

        throw new Error('orderInfoPolling - Ошибка в заказе');
    }

    yield put(updateOrderInfo(newOrderInfo));
}

function* checkOrderStatus(newOrderInfo: IGenericOrderInfo) {
    const {
        trains: {
            order,
            order: {
                error,
                continueWithoutInsurance,
                orderStep,
                isPaymentCalled,
            },
        },
    }: StoreInterface = yield select((state: StoreInterface) => state);

    const {state, id, payment, cancellationReason} = newOrderInfo;

    const trainDetails = getFirstForwardTrainDetails(order);
    const trainService = getFirstTrainService(newOrderInfo);

    if (
        trainService?.trainInfo.insuranceStatus ===
            TrainsInsuranceStatus.CHECKOUT_FAILED &&
        !continueWithoutInsurance &&
        isEmpty(error)
    ) {
        yield put(
            setOrderError({
                type: EOrderErrorType.INSURANCE,
            }),
        );
    }

    if (state === EGenericOrderState.CONFIRMED) {
        const context: ITrainsFilledSearchContext = yield select(
            trainsContextSelector,
        );
        const product = getECommerceProductFromContext(context);

        eCommercePush(ECommerceActionType.PURCHASE, [product], {
            id,
            goal_id: E_COMMERCE_TRAINS_PURCHASE_GOAL_ID,
            revenue: getECommerceRevenue(newOrderInfo),
        });

        if (browserHistory) {
            let checkOrderAuthTries = 0;
            let isSuccessOrderAuth = false;

            while (
                !isSuccessOrderAuth &&
                checkOrderAuthTries++ < MAX_ORDER_AUTH_CHECKS
            ) {
                isSuccessOrderAuth = yield select(
                    (state: StoreInterface) =>
                        state.common.orderAuthorization.createAuthorization
                            .isSuccess,
                );

                if (
                    isSuccessOrderAuth ||
                    checkOrderAuthTries === MAX_ORDER_AUTH_CHECKS
                ) {
                    yield call(() => {
                        browserHistory?.push(trainsURLs.getHappyPageUrl(id));
                    });
                }

                yield delay(ORDER_AUTH_CHECK_DELAY);
            }
        }

        return {isQuerying: false};
    }

    if (state === EGenericOrderState.CANCELLED) {
        switch (cancellationReason) {
            case ECancellationReason.EXPIRED:
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(setTimeoutError());

                break;

            case ECancellationReason.RESERVATION_FAILED:
            case ECancellationReason.CONFIRMATION_FAILED:
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(setOrderCancelledAfterPaidError(newOrderInfo));

                break;
            case ECancellationReason.USER_CANCELLED:
            default:
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(setCancelOrderError());

                break;
        }

        return {isQuerying: false};
    }

    if (state === EGenericOrderState.PAYMENT_FAILED) {
        // ORDER_STATUS_PAYMENT_FAILED возникает и в случаях перехода со вкладки 'Оплата' на 'Подтверждение'
        // или перезагрузке вкладки 'Оплата' из-за скрытия iFrame оплаты
        // В этих случаях мы не должны показывать сообщение об ошибке пользователю

        if (payment?.errorInfo === EOriginalPaymentErrorCode.USER_CANCELLED) {
            return {
                isQuerying: true,
                needToResetPayment: orderStep === ORDER_STEP.PAYMENT,
            };
        }

        yield put(
            setOrderError(
                getPaymentError(newOrderInfo, trainDetails?.tariffCategories),
            ),
        );

        return {isQuerying: false};
    }

    if (
        BEFORE_PAYMENT_STATUSES.includes(normalizeGenericOrderState(state)) &&
        isPaymentCalled
    ) {
        yield put(setPaymentCalled(false));
    }

    return {isQuerying: true};
}

function* pollOrderDetails() {
    try {
        const getLatestOrderInfo = createLatestOrderInfoFetcher();
        let isOrderAuthRequested = false;

        while (true) {
            const state: StoreInterface = yield select();
            const {
                trains: {order},
            } = state;

            const {orderInfo} = order;

            if (!orderInfo || !POLLING_STEPS.includes(order.orderStep)) {
                yield put(cancelOrderPolling());

                return;
            }

            const {id: orderId} = orderInfo;

            const newOrder: IGenericOrderInfo = yield call(
                getLatestOrderInfo,
                orderId,
            );

            // Обнуляем ссылку на оплату
            if (
                newOrder.state === EGenericOrderState.PAYMENT_FAILED &&
                newOrder.payment
            ) {
                newOrder.payment.paymentUrl = '';
            }

            yield dispatchOrderInfo(newOrder);

            const updatedOrderInfo: ReturnType<typeof orderInfoSelector> =
                yield select(orderInfoSelector);

            /**
             * Заказ был отменен и очищен
             * https://st.yandex-team.ru/TRAVELFRONT-3296
             */
            if (!updatedOrderInfo) {
                yield delay(POLLING_INTERVAL);

                continue;
            }

            if (!isOrderAuthRequested) {
                yield put(
                    createOrderAuthorizationActions.request({
                        id: updatedOrderInfo.id,
                        secret: updatedOrderInfo.contactInfo.phone,
                    }),
                );

                isOrderAuthRequested = true;
            }

            const {isQuerying, needToResetPayment} = yield checkOrderStatus(
                updatedOrderInfo,
            );

            if (needToResetPayment) {
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(resetPayment());
            }

            if (
                !isQuerying ||
                isFinalGenericOrderState(updatedOrderInfo.state)
            ) {
                yield put(cancelOrderPolling());
            }

            yield delay(POLLING_INTERVAL);
        }
    } catch (e) {
        logError(
            {
                message: '[YATRAVEL] Ошибка на шаге Ж/д покупки',
                block: 'watchOrderPayment',
            },
            unknownToErrorOrUndefined(e),
        );

        yield put(
            setOrderError(
                isUnknownAxiosError<IOrderError>(e) && e.response?.data
                    ? e.response.data
                    : {type: EOrderErrorType.COMMON},
            ),
        );

        yield put(cancelOrderPolling());

        console.error(e);
    }
}

function* handleOrderPayment() {
    const pollTask: Task = yield fork(pollOrderDetails);

    yield take(getType(cancelOrderPolling));
    yield cancel(pollTask);
}

/**
 * Срабатывает, когда пользователь уходит со страницы заказа.
 *
 * @return {IterableIterator<*>}
 */
function* handleCancelOrder() {
    try {
        const orderInfo: ReturnType<typeof orderInfoSelector> = yield select(
            orderInfoSelector,
        );

        if (orderInfo) {
            yield call(cancelOrderApi, orderInfo);
        }
    } catch (e) {
        /**
         * Пользователь уже ушел со страницы заказа
         * Ошибка отмены брони ему уже не интересна
         */
    }
}

export default function* watchOrderPayment() {
    yield all([
        watchActions(handleOrderPayment, [
            getType(setOrderInfo),
            getType(resumeOrderPolling),
        ]),
        watchActions(handleCancelOrder, [getType(cancelOrder)]),
    ]);
}
