import {call, put, putResolve, select} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';
import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import {setOrderError, setOrderStep} from 'reducers/trains/order/actions/view';
import clearOrder from 'reducers/trains/order/thunk/clearOrder';
import moveToPayment from 'reducers/trains/order/thunk/moveToPayment';
import {setOrderPageStatus} from 'reducers/trains/order/actions/flow';
import logBandit from 'reducers/trains/order/thunk/logBandit';
import {requestAllTrainDetails} from 'reducers/trains/order/thunk/requestAllTrainDetails';

import orderInfoSelector from 'selectors/trains/order/orderInfoSelector';
import orderPageStatusSelector from 'selectors/trains/order/orderPageStatusSelector';
import {orderStepSelector} from 'selectors/trains/order/orderStepSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import {cancelOrder} from 'projects/trains/lib/api/cancelOrder';
import {getTrainsOrderParamsByQuery} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import {logError} from 'utilities/logger/logError';
import {unknownToErrorOrUndefined} from 'utilities/error';

function* handleOrderStepChange() {
    const orderPageStatus: ReturnType<typeof orderPageStatusSelector> =
        yield select(orderPageStatusSelector);
    const orderStep: ReturnType<typeof orderStepSelector> = yield select(
        orderStepSelector,
    );

    if (orderPageStatus === ORDER_PAGE_STATUSES.FETCHING_DATA) {
        return;
    }

    switch (orderStep) {
        case ORDER_STEP.PLACES:
        case ORDER_STEP.PASSENGERS: {
            const orderInfo: ReturnType<typeof orderInfoSelector> =
                yield select(orderInfoSelector);

            if (orderInfo) {
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(clearOrder());

                try {
                    yield call(cancelOrder, orderInfo);
                } catch (e) {
                    if (orderStep === ORDER_STEP.PASSENGERS) {
                        /**
                         * Показывать ошибку о неудаче при отмене брони
                         * имеет смысл только на странице "Ввода данных пассажиров"
                         * https://st.yandex-team.ru/TRAVELFRONT-3091
                         */
                        yield put(
                            setOrderError({
                                type: EOrderErrorType.INTERNAL_ERROR,
                            }),
                        );
                    }
                }

                try {
                    const query = getTrainsOrderParamsByQuery(
                        getQueryByBrowserHistory(),
                    );

                    yield putResolve(
                        // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                        requestAllTrainDetails(query),
                    );

                    yield put(
                        setOrderPageStatus(
                            ORDER_PAGE_STATUSES.READY_TO_INTERACTION,
                        ),
                    );
                } catch (e) {
                    yield put(
                        setOrderError({
                            type: EOrderErrorType.INTERNAL_ERROR,
                        }),
                    );

                    logError(
                        {
                            message: `[YATRAVEL] [TRAINS] Не удалось вернутся на шаг выбора мест/ввода данных пассажира`,
                            block: 'handleOrderStepChange',
                        },
                        unknownToErrorOrUndefined(e),
                    );
                }
            }

            if (orderStep === ORDER_STEP.PASSENGERS) {
                // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
                yield put(logBandit());
            }

            break;
        }
        case ORDER_STEP.PAYMENT: {
            // @ts-ignore redux-saga не понимает, что здесь можно использовать thunk-action
            yield put(moveToPayment());

            break;
        }
    }
}

export default function* watchOrderStepChange() {
    yield watchActions(handleOrderStepChange, [getType(setOrderStep)]);
}
