import {select, call, delay} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';

import {setCount} from 'reducers/trains/order/actions/passengers';
import {setOrderInfo} from 'reducers/trains/order/actions/data';
import {StoreInterface} from 'reducers/storeTypes';
import {
    addPlacesToOrder,
    removePlacesFromOrder,
    setCoach,
    setOrderCoachType,
    setSchemeHasGroupedPlaces,
} from 'reducers/trains/order/actions/trains';

import currentOrderStepDescriptionSelector from 'selectors/trains/order/currentOrderStepDescriptionSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import history from 'utilities/browserHistory/browserHistory';
import {trainsURLs} from 'projects/trains/lib/urls';
import getFirstForwardSegment from 'projects/trains/lib/complexOrder/getFirstForwardSegment';
import getFirstBackwardSegment from 'projects/trains/lib/complexOrder/getFirstBackwardSegment';

/**
 * Функция сохранения состояния покупки в url
 */
function* trackOrder() {
    // Если несколько экшенов одновременно влияют на урл, мы не хотим, чтобы у пользователя моргал урл.
    yield delay(0);

    const {
        trains: {order, context},
    }: StoreInterface = yield select();
    const segment = getFirstForwardSegment(order);
    const backwardSegment = getFirstBackwardSegment(order);
    const orderStepDescription = currentOrderStepDescriptionSelector(
        yield select(),
    );

    if (
        order.orderPageStatus === ORDER_PAGE_STATUSES.FETCHING_DATA ||
        !isFilledTrainsSearchContext(context) ||
        !segment ||
        !history
    ) {
        return;
    }

    const url = trainsURLs.getTrainOrderUrl({
        context,
        segment,
        backwardSegment,
        order,
        orderStepDescription,
    });

    yield call(() => history?.replace(url, {scrollTop: null}));
}

/*
 * Сага следит за влияющими на url экшенами.
 */
export default function* watchOrderUrlRelatedChanges() {
    yield watchActions(trackOrder, [
        getType(setCoach),
        getType(setOrderInfo),
        getType(setOrderCoachType),
        getType(addPlacesToOrder),
        getType(removePlacesFromOrder),
        getType(setSchemeHasGroupedPlaces),
        getType(setCount[PASSENGERS_TYPES.ADULTS]),
        getType(setCount[PASSENGERS_TYPES.CHILDREN]),
        getType(setCount[PASSENGERS_TYPES.BABIES]),
    ]);
}
