import {select, put, call} from 'redux-saga/effects';
import {getType} from 'typesafe-actions';
import {batchActions} from 'redux-batched-actions';
import flatMap from 'lodash/flatMap';
import times from 'lodash/times';

import {PASSENGERS_TYPES} from 'projects/trains/constants/passengersTypes';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';

import {DIRECTIONS} from 'types/common/EDirection';

import {setCount} from 'reducers/trains/order/actions/passengers';
import {setPreviousCountOfPassengersWithPlaces} from 'reducers/trains/order/actions/userChoice';
import {StoreInterface} from 'reducers/storeTypes';
import {
    clearPlaces,
    clearRequirements,
    setPreferredGender,
} from 'reducers/trains/order/actions/trains';

import directionTrainsCountSelector from 'selectors/trains/order/directionTrainsCountSelector';

import watchActions from 'sagas/trains/helpers/watchActions';

import {params} from 'utilities/metrika';
import getUnexpectedStates from 'projects/trains/lib/order/getUnexpectedStates';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {saveTrainsPassengersCount} from 'projects/trains/lib/order/passengers/passengersCountStorage';
import getFirstForwardTrainDetails from 'projects/trains/lib/complexOrder/getFirstForwardTrainDetails';

function* handlePassengersCountChange() {
    const {
        trains: {order},
    }: StoreInterface = yield select();

    if (order.orderPageStatus === ORDER_PAGE_STATUSES.READY_TO_INTERACTION) {
        const unexpectedStates = getUnexpectedStates(order);
        const {passengers, previousCountOfPassengersWithPlaces} = order;
        const trainDetails = getFirstForwardTrainDetails(order);
        const passengersWithPlace = countPassengersWithPlaces(passengers);
        const actions = [];

        if (trainDetails) {
            saveTrainsPassengersCount(passengers, trainDetails.isCppk);
        }

        if (unexpectedStates) {
            yield call(params, {trains: unexpectedStates});
        }

        const directionTrainsCount: ReturnType<
            typeof directionTrainsCountSelector
        > = yield select(directionTrainsCountSelector);

        if (previousCountOfPassengersWithPlaces !== passengersWithPlace) {
            const clearActions = flatMap(DIRECTIONS, direction => {
                const trainCount = directionTrainsCount[direction];

                return flatMap(times(trainCount), index => {
                    const indexAndDirection = {
                        index,
                        direction,
                    };

                    return [
                        clearPlaces({
                            ...indexAndDirection,
                            data: undefined,
                        }),
                        clearRequirements({
                            ...indexAndDirection,
                            data: undefined,
                        }),
                        setPreferredGender({
                            ...indexAndDirection,
                            data: null,
                        }),
                    ];
                });
            });

            actions.push(...clearActions);
        }

        actions.push(
            setPreviousCountOfPassengersWithPlaces(passengersWithPlace),
        );

        yield put(batchActions(actions));
    }
}

export default function* watchPassengersCountChange() {
    yield watchActions(handlePassengersCountChange, [
        getType(setCount[PASSENGERS_TYPES.ADULTS]),
        getType(setCount[PASSENGERS_TYPES.CHILDREN]),
        getType(setCount[PASSENGERS_TYPES.BABIES]),
    ]);
}
