import {
    call,
    put,
    race,
    take,
    select,
    takeLatest,
    CallEffect,
    ForkEffect,
    RaceEffect,
    TakeEffect,
} from 'redux-saga/effects';
import {getType, ActionType} from 'typesafe-actions';
import {SagaIterator} from 'redux-saga';

import {
    ITrainsPriceCalendarRequestParams,
    ITrainsPriceCalendarResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsPriceCalendar';
import {EDirection} from 'types/common/EDirection';
import {ITrainsSearchFormPointField} from 'projects/trains/components/SearchForm/types';
import {ITrainsStartPriceCalendarActionPayload} from 'reducers/trains/priceCalendar/types';

import {
    trainsFetchPriceCalendarActions,
    trainsPriceCalendarClearAction,
} from 'reducers/trains/priceCalendar/actions';

import searchFormSelector from 'selectors/trains/searchForm/searchFormSelector';

import {logDebug} from 'utilities/logger/logDebug';
import {checkYandexMollyVisit} from 'utilities/checkYandexMollyVisit/checkYandexMollyVisit';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';

import {trainsSearchProvider} from 'serviceProvider';

/* Helpers */
function* getPriceCalendarParams(
    payload: ITrainsStartPriceCalendarActionPayload,
): SagaIterator<ITrainsPriceCalendarRequestParams | undefined> {
    const {pointFrom, pointTo} = payload;
    const queryParams = getQueryByBrowserHistory();

    return {
        pointFrom,
        pointTo,
        isBot: checkYandexMollyVisit(queryParams),
    };
}

const getFromAndToFields = ({
    fromField,
    toField,
}: {
    fromField: ITrainsSearchFormPointField;
    toField: ITrainsSearchFormPointField;
}): {
    fromFieldKey?: string;
    fromFieldSlug?: string;
    fromFieldFullTitle?: string;
    toFieldKey?: string;
    toFieldSlug?: string;
    toFieldFullTitle?: string;
} => {
    const fromSelectedValue = fromField.selectedValue;
    const toSelectedValue = toField.selectedValue;

    if (
        typeof fromSelectedValue !== 'boolean' &&
        typeof toSelectedValue !== 'boolean'
    ) {
        return {
            fromFieldKey: fromSelectedValue.pointKey,
            fromFieldSlug: fromSelectedValue.slug,
            fromFieldFullTitle: fromSelectedValue.fullTitle,
            toFieldKey: toSelectedValue.pointKey,
            toFieldSlug: toSelectedValue.slug,
            toFieldFullTitle: toSelectedValue.fullTitle,
        };
    }

    return {};
};

function* fetchPriceCalendar(payload: ITrainsStartPriceCalendarActionPayload) {
    try {
        const requestParams: ITrainsPriceCalendarRequestParams = yield call(
            getPriceCalendarParams,
            payload,
        );
        const {forward, backward}: ITrainsPriceCalendarResponse = yield call(
            trainsSearchProvider.provider().priceCalendar,
            requestParams,
        );

        yield put(
            trainsFetchPriceCalendarActions.success({
                [EDirection.FORWARD]: forward,
                [EDirection.BACKWARD]: backward,
            }),
        );
    } catch (e) {
        const {fromField, toField}: ReturnType<typeof searchFormSelector> =
            yield select(searchFormSelector);
        const {pointFrom, pointTo} = payload;

        logDebug({
            message:
                '[YATRAVEL][TRAINS][PRICE_CALENDAR] Ошибка получения данных с ценами в календаре',
            block: 'trainsPriceCalendar',
            additional: {
                pointFrom,
                pointTo,
                ...getFromAndToFields({fromField, toField}),
            },
        });

        yield put(trainsFetchPriceCalendarActions.failure());
    }
}

function* priceCalendarWorker(
    action: ActionType<typeof trainsFetchPriceCalendarActions.request>,
): Generator<RaceEffect<TakeEffect | CallEffect>, void, void> {
    yield race({
        startAction: call(fetchPriceCalendar, action.payload),
        stopAction: take(getType(trainsPriceCalendarClearAction)),
    });
}

export default function* watchPriceCalendar(): Generator<
    ForkEffect,
    void,
    void
> {
    yield takeLatest(
        getType(trainsFetchPriceCalendarActions.request),
        priceCalendarWorker,
    );
}
