import {createSelector} from 'reselect';

import {
    IFlightsToDynamicsBlock,
    IAviaFlightsToResponse,
    IFlightsToSearchFormBlock,
    TAviaFlightsToBlock,
} from 'types/avia/landing/flightsTo/IAviaFlightsTo';
import {
    ELandingBlockType,
    IAviaSeoInfoSchemaOrg,
    ILandingSectionTextBlock,
    IBreadCrumbsBlock,
    IClosestCitiesBlock,
    ICityDirectionsBlock,
} from 'types/avia/landing/IAviaLanding';
import {ISeoInfo} from 'types/hotels/common/ISeoInfo';

import nonceSelector from 'selectors/common/nonceSelector';
import {getAviaContext, getAviaFlightsTo} from 'selectors/avia/aviaSelectors';

type TBlocksType =
    | ILandingSectionTextBlock
    | IClosestCitiesBlock
    | ICityDirectionsBlock;

interface IFlightsToDataSelector {
    seoInfo: ISeoInfo<IAviaSeoInfoSchemaOrg> | undefined;
    breadcrumbs: IBreadCrumbsBlock | undefined;
    searchForm: IFlightsToSearchFormBlock | undefined;
    dynamicsBlock: IFlightsToDynamicsBlock | undefined;
    blocks: TBlocksType[] | undefined;
    navigation: string[] | undefined;
    nonce: ReturnType<typeof nonceSelector>;
    aviaContext: ReturnType<typeof getAviaContext>;
    isFetched: boolean;
    isError: boolean;
    isNotFound: boolean;
}

type TGetAviaFlightsToBlockByType<
    Block extends TAviaFlightsToBlock,
    Type extends ELandingBlockType,
> = Block extends {type: Type} ? Block : never;

export function getFlightsToContentBlock<T extends ELandingBlockType>(
    type: T,
    data?: IAviaFlightsToResponse,
): TGetAviaFlightsToBlockByType<TAviaFlightsToBlock, T> | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks.find(
            block => block.type === type,
        ) as TGetAviaFlightsToBlockByType<TAviaFlightsToBlock, T>;
    }

    return undefined;
}

function filterContentBlocks(
    data?: IAviaFlightsToResponse,
): TBlocksType[] | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks.filter(
            block =>
                block.type !== ELandingBlockType.dynamicsBlock &&
                block.type !== ELandingBlockType.searchFormBlock &&
                block.type !== ELandingBlockType.breadCrumbsBlock,
        ) as TBlocksType[];
    }

    return undefined;
}

function getNavigation(data?: IAviaFlightsToResponse): string[] | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks
            .map(block =>
                'navigationTitle' in block.data && block.data.navigationTitle
                    ? block.data.navigationTitle
                    : '',
            )
            .filter(block => block !== '');
    }

    return undefined;
}

const flightsToSelector = createSelector(getAviaFlightsTo, aviaFlightsTo => ({
    seoInfo: aviaFlightsTo.data?.seoInfo,
    breadcrumbs: getFlightsToContentBlock(
        ELandingBlockType.breadCrumbsBlock,
        aviaFlightsTo.data,
    ),
    searchForm: getFlightsToContentBlock(
        ELandingBlockType.searchFormBlock,
        aviaFlightsTo.data,
    ),
    blocks: filterContentBlocks(aviaFlightsTo.data),
    dynamicsBlock: getFlightsToContentBlock(
        ELandingBlockType.dynamicsBlock,
        aviaFlightsTo.data,
    ),
    navigation: getNavigation(aviaFlightsTo.data),
    isError: aviaFlightsTo.isError,
    isNotFound: aviaFlightsTo.isNotFound,
    isFetched: aviaFlightsTo.isFetched,
}));

export default createSelector(
    [flightsToSelector, nonceSelector, getAviaContext],
    (flightsToData, nonce, aviaContext): IFlightsToDataSelector => ({
        ...flightsToData,
        nonce,
        aviaContext,
    }),
);
