import {createSelector} from 'reselect';

import {
    IAviaRouteResponse,
    TAviaRouteBlock,
    IRouteDynamicsBlock,
    IRouteSearchFormBlock,
} from 'types/avia/landing/route/IAviaRoute';
import {
    ELandingBlockType,
    IAviaSeoInfoSchemaOrg,
    ILandingSectionTextBlock,
    IBreadCrumbsBlock,
    IClosestCitiesBlock,
    IReturnTicketBlock,
    IAviaCompaniesBlock,
} from 'types/avia/landing/IAviaLanding';
import {ISeoInfo} from 'types/hotels/common/ISeoInfo';

import nonceSelector from 'selectors/common/nonceSelector';
import {getAviaRoute} from 'selectors/avia/aviaSelectors';

type TBlocksType =
    | ILandingSectionTextBlock
    | IAviaCompaniesBlock
    | IClosestCitiesBlock
    | IReturnTicketBlock;

interface IRouteDataSelector {
    seoInfo: ISeoInfo<IAviaSeoInfoSchemaOrg> | undefined;
    breadcrumbs: IBreadCrumbsBlock | undefined;
    searchForm: IRouteSearchFormBlock | undefined;
    dynamicsBlock: IRouteDynamicsBlock | undefined;
    blocks: TBlocksType[] | undefined;
    navigation: string[] | undefined;
    nonce: ReturnType<typeof nonceSelector>;
    isFetched: boolean;
    isError: boolean;
    isNotFound: boolean;
}

type TGetAviaRouteBlockByType<
    Block extends TAviaRouteBlock,
    Type extends ELandingBlockType,
> = Block extends {type: Type} ? Block : never;

export function getRouteContentBlock<T extends ELandingBlockType>(
    type: T,
    data?: IAviaRouteResponse,
): TGetAviaRouteBlockByType<TAviaRouteBlock, T> | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks.find(
            block => block.type === type,
        ) as TGetAviaRouteBlockByType<TAviaRouteBlock, T>;
    }

    return undefined;
}

function filterContentBlocks(
    data?: IAviaRouteResponse,
): TBlocksType[] | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks.filter(
            block =>
                block.type !== ELandingBlockType.dynamicsBlock &&
                block.type !== ELandingBlockType.searchFormBlock &&
                block.type !== ELandingBlockType.breadCrumbsBlock,
        ) as TBlocksType[];
    }

    return undefined;
}

function getNavigation(data?: IAviaRouteResponse): string[] | undefined {
    if (data && data.blocks.length > 0) {
        return data.blocks
            .map(block =>
                'navigationTitle' in block.data && block.data.navigationTitle
                    ? block.data.navigationTitle
                    : '',
            )
            .filter(block => block !== '');
    }

    return undefined;
}

const routeSelector = createSelector(getAviaRoute, aviaRoute => ({
    seoInfo: aviaRoute.data?.seoInfo,
    breadcrumbs: getRouteContentBlock(
        ELandingBlockType.breadCrumbsBlock,
        aviaRoute.data,
    ),
    searchForm: getRouteContentBlock(
        ELandingBlockType.searchFormBlock,
        aviaRoute.data,
    ),
    blocks: filterContentBlocks(aviaRoute.data),
    dynamicsBlock: getRouteContentBlock(
        ELandingBlockType.dynamicsBlock,
        aviaRoute.data,
    ),
    navigation: getNavigation(aviaRoute.data),
    isError: aviaRoute.isError,
    isNotFound: aviaRoute.isNotFound,
    isFetched: aviaRoute.isFetched,
}));

export default createSelector(
    [routeSelector, nonceSelector],
    (routeData, nonce): IRouteDataSelector => ({
        ...routeData,
        nonce,
    }),
);
