import {createSelector} from 'reselect';
import {groupBy, mapValues, minBy, partition} from 'lodash';
import uniqBy from 'lodash/uniqBy';

import {isNotNullable} from 'types/utilities';

import {StoreInterface} from 'reducers/storeTypes';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {
    getAviaContext,
    qidSelector,
    useOrderDataSelector,
} from 'selectors/avia/aviaSelectors';
import {sortResultVariants} from 'selectors/avia/utils/sortResultVariants';
import {priceComparatorSelector} from 'selectors/common/currenciesSelector';

import {
    EAviaOfferType,
    getAviaOfferType,
} from 'projects/avia/lib/search/offerType';

export interface IVariantsByTariff {
    cheapest: IResultAviaVariant | null;
    cheapestWithBaggage: IResultAviaVariant | null;
    withBaggage: {
        freeRefund: IResultAviaVariant[];
        chargedRefund: IResultAviaVariant[];
        noOrUnknownRefund: IResultAviaVariant[];
    };
    withoutBaggage: {
        freeRefund: IResultAviaVariant[];
        chargedRefund: IResultAviaVariant[];
        noOrUnknownRefund: IResultAviaVariant[];
    };
}

export interface IAviaOrderOffers {
    offerType?: EAviaOfferType;
    cheapest: IResultAviaVariant;
    avia: IResultAviaVariant[];
    other: IResultAviaVariant[];
    variantsByTariff: IVariantsByTariff;
}

const hasGoodPriceSelector = (state: StoreInterface): boolean =>
    state.avia.aviaOrder.hasGoodPrice;

const contSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaSearch.results.cont,
    (state: StoreInterface) => state.avia.aviaOrder.cont,
    (useOrderData, searchCont, orderCont) =>
        useOrderData ? orderCont : searchCont,
);

const baggageFilterSelector = (
    state: StoreInterface,
    props: {withBaggage?: boolean},
): boolean => props.withBaggage ?? state.avia.aviaOrder.withBaggage;

const badPartnersSelector = (state: StoreInterface): string[] =>
    state.avia.aviaOrder.badPartners;

const progressSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaSearch.results.progress,
    (state: StoreInterface) => state.avia.aviaOrder.progress,
    (useOrderData, searchProgress, orderProgress) =>
        useOrderData ? orderProgress : searchProgress,
);

const searchStatusSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaSearch.results.searchIsCompleted,
    (state: StoreInterface) => state.avia.aviaOrder.partnersDataWereFetched,
    (useOrderData, searchStatus, orderStatus) =>
        useOrderData ? orderStatus : searchStatus,
);

const referenceSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaOrder.reference,
    (state: StoreInterface) => state.avia.aviaSearch.results.reference,
    (useOrderData, orderReference, searchReference) =>
        useOrderData ? orderReference : searchReference,
);

const variantsSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaOrder.variants,
    (state: StoreInterface) => state.avia.aviaSearch.results.variants,
    (useOrderData, orderVariants, searchVariants) =>
        useOrderData ? orderVariants : searchVariants,
);

const variantGroupsSelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaOrder.variantGroups,
    (state: StoreInterface) => state.avia.aviaSearch.results.variantGroups,
    (useOrderData, orderVariantGroups, searchVariantGroups) =>
        useOrderData ? orderVariantGroups : searchVariantGroups,
);

const variantKeySelector = createSelector(
    useOrderDataSelector,
    (state: StoreInterface) => state.avia.aviaOrder.variantKey,
    (state: StoreInterface, props: {variantKey?: string}): string | undefined =>
        props?.variantKey,
    (useOrderData, orderVariantKey, searchVariantKey) =>
        useOrderData ? orderVariantKey : searchVariantKey,
);

const rawVariantsSelector = createSelector(
    variantKeySelector,
    variantsSelector,
    variantGroupsSelector,
    (variantKey, variants, variantGroups) =>
        variantKey && variantGroups[variantKey]
            ? variantGroups[variantKey].map(tag => variants[tag])
            : null,
);

export const variantSelector = createSelector(rawVariantsSelector, variants => {
    if (variants && variants.length > 1) {
        return variants.filter(
            ({
                price: {
                    partner: {code},
                },
            }) => code !== 'aviaFront',
        );
    }

    return variants || [];
});

// селектор определяет есть ли среди вариантов - багаж с бонусами от Плюса
// и если есть - возвращает их количество
const defaultPlusPointsSelector = createSelector(variantSelector, variants => {
    const defaultPlusPoints = variants.find(
        item => item.hasBaggage && isNotNullable(item.price?.plusPoints),
    );

    return defaultPlusPoints ? defaultPlusPoints.price.plusPoints : undefined;
});

export const filteredVariantSelector = createSelector(
    variantSelector,
    baggageFilterSelector,
    (variants, withBaggage) => {
        const variantsWithBaggage = variants.filter(
            ({hasBaggage}) => hasBaggage,
        );
        const baggageSelectIsAvailable =
            variantsWithBaggage.length &&
            variantsWithBaggage.length !== variants.length;

        return Object.values(
            mapValues(
                groupBy(
                    baggageSelectIsAvailable && withBaggage
                        ? variantsWithBaggage
                        : variants,
                    item => item.price.partner.code,
                ),
                grouped =>
                    minBy(
                        grouped,
                        item => item.price.tariff.value,
                    ) as IResultAviaVariant,
            ),
        );
    },
);

export const cheapestVariantSelector = createSelector(
    filteredVariantSelector,
    priceComparatorSelector,
    (variants, priceComparator) => {
        const sortedVariants = sortResultVariants(variants, priceComparator);
        const [cheapest = undefined] = sortedVariants;

        return cheapest;
    },
);

const baggageSelector = createSelector(
    variantSelector,
    baggageFilterSelector,
    (allVariants, withBaggage) => {
        const variantsWithBaggage = allVariants.filter(item => item.hasBaggage);
        const baggagePrice =
            (variantsWithBaggage.length &&
                variantsWithBaggage.length !== allVariants.length &&
                minBy(
                    variantsWithBaggage.map(({price}) => price && price.tariff),
                    'value',
                )) ||
            null;

        return {
            withBaggage,
            baggagePrice,
        };
    },
);

export const sortedVariantsSelector = createSelector(
    variantSelector,
    priceComparatorSelector,
    badPartnersSelector,
    (dirtyVariants, priceComparator, badPartners) => {
        const variants = dirtyVariants.filter(
            ({price}) =>
                price.partner.code !== 'aviaFront' &&
                !badPartners.includes(price.partner.code),
        );

        return uniqBy(
            sortResultVariants(variants, priceComparator),
            ({price}) => `${price.partner.code}${price.fareFamiliesHash}`,
        );
    },
);

/**
 * Временный селектор на время эксперимента https://st.yandex-team.ru/TRAVELFRONT-5615
 * Частично дублирует offersSelector.
 */
export const tariffGroupsSelector = createSelector(
    sortedVariantsSelector,
    sortedVariants => {
        return sortedVariants.reduce<IVariantsByTariff>(
            (groups, variant) => {
                const baggageGroup = variant.price.hasBaggage
                    ? groups.withBaggage
                    : groups.withoutBaggage;

                if (variant.price.hasFreeRefund) {
                    baggageGroup.freeRefund.push(variant);
                } else if (variant.price.hasChargedRefund) {
                    baggageGroup.chargedRefund.push(variant);
                } else {
                    baggageGroup.noOrUnknownRefund.push(variant);
                }

                if (groups.cheapest === null) {
                    groups.cheapest = variant;
                }

                if (groups.cheapestWithBaggage === null) {
                    if (variant.price.hasBaggage) {
                        groups.cheapestWithBaggage = variant;
                    }
                }

                return groups;
            },
            {
                cheapest: null,
                cheapestWithBaggage: null,
                withBaggage: {
                    freeRefund: [],
                    chargedRefund: [],
                    noOrUnknownRefund: [],
                },
                withoutBaggage: {
                    freeRefund: [],
                    chargedRefund: [],
                    noOrUnknownRefund: [],
                },
            },
        );
    },
);

export const offersSelector = createSelector(
    filteredVariantSelector,
    priceComparatorSelector,
    badPartnersSelector,
    tariffGroupsSelector,
    (
        dirtyVariants,
        priceComparator,
        badPartners,
        variantsByTariff,
    ): IAviaOrderOffers | null => {
        const variants = (dirtyVariants || []).filter(
            ({
                price: {
                    partner: {code},
                },
            }) => code !== 'aviaFront' && !badPartners.includes(code),
        );

        if (!variants.length) {
            return null;
        }

        const offerType = getAviaOfferType(variants);
        const sortedVariants = sortResultVariants(variants, priceComparator);
        const cheapest = sortedVariants.shift() as IResultAviaVariant;

        const [avia, other] = partition(
            sortedVariants,
            variant => variant.price.fromCompany,
        );

        return {
            variantsByTariff,
            offerType,
            cheapest,
            avia,
            other,
        };
    },
);

const errorSelector = (state: StoreInterface) => state.avia.aviaOrder.error;

export const searchIsCompleteSelector = createSelector(
    contSelector,
    searchStatusSelector,
    (cont, partnersDataWereFetched) => cont === null && partnersDataWereFetched,
);

export const aviaOrderSelectors = {
    searchIsComplete: searchIsCompleteSelector,
    qid: qidSelector,
    offers: offersSelector,
    baggage: baggageFilterSelector,
    defaultPlusPoints: defaultPlusPointsSelector,
    badPartners: badPartnersSelector,
    useOrderData: useOrderDataSelector,
    reference: referenceSelector,
};

export const aviaOrderExtendedSelector = createSelector(
    sortedVariantsSelector,
    sortedVariants => ({sortedVariants}),
);

export default createSelector(
    qidSelector,
    getAviaContext,
    useOrderDataSelector,
    hasGoodPriceSelector,
    searchIsCompleteSelector,
    progressSelector,
    referenceSelector,
    offersSelector,
    baggageSelector,
    errorSelector,
    defaultPlusPointsSelector,
    cheapestVariantSelector,
    (
        qid,
        context,
        useOrderData,
        hasGoodPrice,
        searchIsCompleted,
        progress,
        reference,
        offers,
        {withBaggage, baggagePrice},
        error,
        defaultPlusPoints,
        cheapestVariant,
    ) => ({
        qid,
        context,
        useOrderData,
        hasGoodPrice,
        searchIsCompleted,
        progress,
        offers,
        reference,
        withBaggage,
        baggagePrice,
        error,
        defaultPlusPoints,
        cheapestVariant,
    }),
);
