import {createSelector} from 'reselect';

import {ISettlement} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {StoreInterface} from 'reducers/storeTypes';

import {getDenormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';

import {SegmentStationsMaps} from 'projects/avia/lib/search/getStationMaps';
import {aviaFilterBy} from 'projects/avia/lib/search/filters/filterVariants';

import {validateValue} from './validateValue';
import {getInitialViewData} from './getInitialViewData';

export interface SegmentStationViewState {
    arrival: number;
    departure: number;
    transfers: number[];
}

export interface DisabledSegmentStationsViewState {
    arrival: Record<number, boolean>;
    departure: Record<number, boolean>;
    transfers: Record<number, boolean>;
}

export interface AirportsFilterSettlements {
    from: ISettlement;
    to: ISettlement;
}

export interface AirportsFilterViewData {
    stations: SegmentStationsMaps[];
    disabledStations: DisabledSegmentStationsViewState[];
    stationsByTag: Record<string, SegmentStationViewState[]>;
    settlements: AirportsFilterSettlements[];
}

const valueSelector = (state: StoreInterface) =>
    state.avia.aviaSearch.results.filters.airports;

const initialViewDataSelector = createSelector(
    getDenormalizedVariants,
    getInitialViewData,
);

const validValueSelector = createSelector(
    valueSelector,
    initialViewDataSelector,
    validateValue,
);

const filteredSelector = createSelector(
    validValueSelector,
    getDenormalizedVariants,
    aviaFilterBy.airports,
);

const directionsIsActiveSelector = createSelector(validValueSelector, value =>
    value.some(
        airports =>
            airports.arrival.length !== 0 || airports.departure.length !== 0,
    ),
);

const transfersIsActiveSelector = createSelector(validValueSelector, value =>
    value.some(airports => airports.transfers.length !== 0),
);

const directionsSelectedCountSelector = createSelector(
    validValueSelector,
    value => {
        return value.reduce((count, airports) => {
            return count + airports.arrival.length + airports.departure.length;
        }, 0);
    },
);

const transfersSelectedCountSelector = createSelector(
    validValueSelector,
    value => {
        return value.reduce((count, airports) => {
            return count + airports.transfers.length;
        }, 0);
    },
);

export const aviaAirportsFilterSelector = {
    key: 'airports',
    directionsIsActive: directionsIsActiveSelector,
    transfersIsActive: transfersIsActiveSelector,
    directionsSelectedCount: directionsSelectedCountSelector,
    transfersSelectedCount: transfersSelectedCountSelector,
    filtered: filteredSelector,
    initialViewData: initialViewDataSelector,
    value: validValueSelector,
};
