import {
    IAviaSearchResultsFilters,
    TAviaSearchAirportsFilterKeys,
} from 'reducers/avia/search/results/filters/reducer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {
    AirportsFilterViewData,
    DisabledSegmentStationsViewState,
    SegmentStationViewState,
} from './index';

export function enreachViewData(
    variant: IResultAviaVariant,
    filtered: Record<string, IResultAviaVariant>,
    filteredByOthers: Record<string, boolean>,
    value: IAviaSearchResultsFilters['airports'],
    viewData: AirportsFilterViewData,
) {
    const {tag, route} = variant;
    const {disabledStations, stationsByTag} = viewData;
    const stationsForVariant = stationsByTag[tag];
    const isFilteredByOthers = Boolean(filteredByOthers[tag]);
    const isFiltered = Boolean(filtered[tag]);

    for (let segmentIdx = 0; segmentIdx < route.length; segmentIdx++) {
        const segmentStationsForVariant = stationsForVariant[segmentIdx];
        if (!segmentStationsForVariant) {
            return;
        }

        let disabledSegmentStations = disabledStations[segmentIdx];
        if (!disabledSegmentStations) {
            disabledSegmentStations = disabledStations[segmentIdx] = {
                arrival: {},
                departure: {},
                transfers: {},
            };
        }

        updateDisabled({
            direction: 'arrival',
            disabledSegmentStations,
            isFiltered,
            isFilteredByOthers,
            segmentIdx,
            stationId: segmentStationsForVariant.arrival,
            stationsForVariant,
            value,
        });

        updateDisabled({
            direction: 'departure',
            disabledSegmentStations,
            isFiltered,
            isFilteredByOthers,
            segmentIdx,
            stationId: segmentStationsForVariant.departure,
            stationsForVariant,
            value,
        });

        for (let j = 0; j < segmentStationsForVariant.transfers.length; j++) {
            updateDisabled({
                direction: 'transfers',
                disabledSegmentStations,
                isFiltered,
                isFilteredByOthers,
                segmentIdx,
                stationId: segmentStationsForVariant.transfers[j],
                stationsForVariant,
                value,
            });
        }
    }
}

type UpdateDisabledParams = {
    stationId: number;
    segmentIdx: number;
    direction: TAviaSearchAirportsFilterKeys;
    isFilteredByOthers: boolean;
    isFiltered: boolean;
    disabledSegmentStations: DisabledSegmentStationsViewState;
    value: IAviaSearchResultsFilters['airports'];
    stationsForVariant: SegmentStationViewState[];
};

function updateDisabled({
    stationId,
    segmentIdx,
    direction,
    isFilteredByOthers,
    isFiltered,
    disabledSegmentStations,
    stationsForVariant,
    value,
}: UpdateDisabledParams) {
    const disabledDirStations = disabledSegmentStations[direction];
    const noData = typeof disabledDirStations[stationId] === 'undefined';

    const checkRequired = noData || disabledDirStations[stationId];
    const disabled =
        isFilteredByOthers ||
        (isFiltered &&
            checkRequired &&
            shouldDisable(segmentIdx, direction, stationsForVariant, value));

    if (noData) {
        disabledDirStations[stationId] = disabled;
    } else if (!disabled && disabledDirStations[stationId]) {
        disabledDirStations[stationId] = false;
    }
}

export function shouldDisable(
    segmentIdx: number,
    direction: TAviaSearchAirportsFilterKeys,
    variantStations: SegmentStationViewState[],
    filterValue: IAviaSearchResultsFilters['airports'],
): boolean {
    for (let i = 0; i < filterValue.length; i++) {
        const segmentFilter = filterValue[i];
        const segmentStations = variantStations[i];
        if (!segmentStations || !segmentFilter) {
            continue;
        }

        if (
            !(segmentIdx === i && direction === 'departure') &&
            segmentFilter.departure.length > 0
        ) {
            if (!segmentFilter.departure.includes(segmentStations.departure)) {
                return true;
            }
        }

        if (
            !(segmentIdx === i && direction === 'arrival') &&
            segmentFilter.arrival.length > 0
        ) {
            if (!segmentFilter.arrival.includes(segmentStations.arrival)) {
                return true;
            }
        }

        if (
            !(segmentIdx === i && direction === 'transfers') &&
            segmentFilter.transfers.length > 0
        ) {
            for (let j = 0; j < segmentStations.transfers.length; j++) {
                const stationId = segmentStations.transfers[j];
                if (!segmentFilter.transfers.includes(stationId)) {
                    return true;
                }
            }
        }
    }

    return false;
}

export function aviaUpdateAirportsViewData(
    variants: IResultAviaVariant[],
    filtered: Record<string, IResultAviaVariant>,
    filteredByOthers: Record<string, boolean>,
    value: IAviaSearchResultsFilters['airports'],
    viewData: AirportsFilterViewData,
): AirportsFilterViewData {
    for (let i = 0; i < variants.length; i++) {
        enreachViewData(
            variants[i],
            filtered,
            filteredByOthers,
            value,
            viewData,
        );
    }

    return viewData;
}
