import {createSelector} from 'reselect';

import {StoreInterface} from 'reducers/storeTypes';
import {EAviaSearchTimeFilter} from 'reducers/avia/search/results/filters/reducer';

import {getDenormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';
import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';

import {Flag, toArray} from 'utilities/flags/flags';
import {aviaFilterBy} from 'projects/avia/lib/search/filters/filterVariants';

export interface ITimeFilterViewData {
    disabled: {
        arrival: Flag<EAviaSearchTimeFilter>;
        departure: Flag<EAviaSearchTimeFilter>;
    }[];
    segmentsLength: number;
}

const valueSelector = (state: StoreInterface) =>
    state.avia.aviaSearch.results.filters.time;

const initialViewDataSelector = createSelector(
    getDenormalizedVariants,
    variants => {
        const [first] = variants;
        const segmentsLength =
            first &&
            first.route.filter((s: IResultAviaFlight[]) => s.length > 0).length;

        return {
            disabled: [],
            segmentsLength,
        };
    },
);

const validValueSelector = createSelector(
    valueSelector,
    initialViewDataSelector,
    (value, {segmentsLength}) => value.slice(0, segmentsLength),
);

const filteredSelector = createSelector(
    validValueSelector,
    getDenormalizedVariants,
    aviaFilterBy.time,
);

const isActiveSelector = createSelector(validValueSelector, value =>
    value.some(time => time.arrival !== 0 || time.departure !== 0),
);

const selectedCountSelector = createSelector(validValueSelector, value => {
    return value.reduce((count, time) => {
        return (
            count +
            toArray(time.arrival).length +
            toArray(time.departure).length
        );
    }, 0);
});

export const aviaTimeFilterSelector = {
    key: 'time',
    isActive: isActiveSelector,
    selectedCount: selectedCountSelector,
    filtered: filteredSelector,
    initialViewData: initialViewDataSelector,
    value: validValueSelector,
};
