import {
    EAviaSearchTimeFilter,
    IAviaSearchTimeFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {ITimeFilterViewData} from 'selectors/avia/search/filters/timeFilter/index';

import {createFlags, disable, join} from 'utilities/flags/flags';
import {isPassingTimeFilters} from 'projects/avia/lib/search/filters/isPassingTimeFilters';

const DAY_TIMES_ALL = join<EAviaSearchTimeFilter>([
    createFlags(EAviaSearchTimeFilter.MORNING),
    createFlags(EAviaSearchTimeFilter.DAY),
    createFlags(EAviaSearchTimeFilter.EVENING),
    createFlags(EAviaSearchTimeFilter.NIGHT),
]);

const getInitDisabledState = () => ({
    arrival: DAY_TIMES_ALL,
    departure: DAY_TIMES_ALL,
});

function getRejectedFilter(
    segmentIdx: number,
    value: IAviaSearchTimeFilter[],
    type: 'arrival' | 'departure',
) {
    return Array.from(value, (val, idx) => {
        if (idx === segmentIdx) {
            return {
                ...val,
                [type]: 0,
            };
        }
        return val;
    });
}

export const enreachViewData = (
    variant: IResultAviaVariant,
    filteredByOthers: Record<string, boolean>,
    viewData: ITimeFilterViewData,
    filtered: Record<string, IResultAviaVariant>,
    filtersRejectedByDeparture: IAviaSearchTimeFilter[][],
    filtersRejectedByArrival: IAviaSearchTimeFilter[][],
) => {
    if (filteredByOthers[variant.tag]) {
        return;
    }

    const {disabled} = viewData;
    const isFiltered = filtered[variant.tag];

    variant.route.forEach((segment, segmentIdx) => {
        if (segment.length === 0) {
            return;
        }

        let disabledState = disabled[segmentIdx];

        if (!disabledState) {
            disabledState = disabled[segmentIdx] = getInitDisabledState();
        }

        if (
            !isFiltered ||
            isPassingTimeFilters(variant, filtersRejectedByArrival[segmentIdx])
        ) {
            disabledState.arrival = disable(
                disabledState.arrival,
                segment[segment.length - 1].arrivalTimeOfDay,
            );
        }

        if (
            !isFiltered ||
            isPassingTimeFilters(
                variant,
                filtersRejectedByDeparture[segmentIdx],
            )
        ) {
            disabledState.departure = disable(
                disabledState.departure,
                segment[0].departureTimeOfDay,
            );
        }
    });
};

function getFiltersRejectedByArrival(
    value: IAviaSearchTimeFilter[],
): IAviaSearchTimeFilter[][] {
    return Array.from(value, (_val, index) =>
        getRejectedFilter(index, value, 'arrival'),
    );
}

function getFiltersRejectedByDeparture(
    value: IAviaSearchTimeFilter[],
): IAviaSearchTimeFilter[][] {
    return Array.from(value, (_val, index) =>
        getRejectedFilter(index, value, 'departure'),
    );
}

export function aviaUpdateTimeViewData(
    variants: IResultAviaVariant[],
    filteredByOthers: Record<string, boolean>,
    viewData: ITimeFilterViewData,
    filtered: Record<string, IResultAviaVariant>,
    value: IAviaSearchTimeFilter[],
) {
    const filtersRejectedByArrival = getFiltersRejectedByArrival(value);
    const filtersRejectedByDeparture = getFiltersRejectedByDeparture(value);

    for (let i = 0; i < variants.length; i++) {
        enreachViewData(
            variants[i],
            filteredByOthers,
            viewData,
            filtered,
            filtersRejectedByDeparture,
            filtersRejectedByArrival,
        );
    }

    return viewData;
}
