import {createSelector} from 'reselect';

import {EAviaSearchTransferFilter} from 'reducers/avia/search/results/filters/reducer';
import {StoreInterface} from 'reducers/storeTypes';

import {getDenormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';
import {validateValue} from 'selectors/avia/search/filters/transferFilter/validateValue';
import {getInitialViewData} from 'selectors/avia/search/filters/transferFilter/getInitialViewData';

import {Flag, isEnabled, toArray} from 'utilities/flags/flags';
import {aviaFilterBy} from 'projects/avia/lib/search/filters/filterVariants';

export interface TransferFilterViewData {
    disabled: Flag<EAviaSearchTransferFilter>;
    minDuration: number;
    maxDuration: number;
    hasTransfers: boolean;
}

const valueSelector = (state: StoreInterface) =>
    state.avia.aviaSearch.results.filters.transfer;

const initialViewDataSelector = createSelector(
    getDenormalizedVariants,
    getInitialViewData,
);

const validValueSelector = createSelector(
    valueSelector,
    initialViewDataSelector,
    validateValue,
);

const filteredSelector = createSelector(
    validValueSelector,
    getDenormalizedVariants,
    aviaFilterBy.transfer,
);

const isActiveSelector = createSelector(
    validValueSelector,
    initialViewDataSelector,
    ({value, range}, {minDuration, maxDuration}) => {
        if (value !== 0) {
            return true;
        }

        if (range !== null) {
            return range[0] !== minDuration || range[1] !== maxDuration;
        }

        return false;
    },
);

const selectedCountSelector = createSelector(
    validValueSelector,
    initialViewDataSelector,
    ({value, range}, {minDuration, maxDuration}) => {
        let count = toArray(value).length;
        const isNoTransfers = isEnabled(
            value,
            EAviaSearchTransferFilter.NO_TRANSFERS,
        );

        if (
            !isNoTransfers &&
            range !== null &&
            (range[0] !== minDuration || range[1] !== maxDuration)
        ) {
            ++count;
        }

        return count;
    },
);

export const aviaTransferFilterSelector = {
    key: 'transfer',
    isActive: isActiveSelector,
    selectedCount: selectedCountSelector,
    filtered: filteredSelector,
    initialViewData: initialViewDataSelector,
    value: validValueSelector,
};
