import {
    IAviaSearchTransferFilter,
    EAviaSearchTransferFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {TransferFilterViewData} from 'selectors/avia/search/filters/transferFilter/index';
import {getMinMaxTransfersDurationForRoute} from 'selectors/avia/search/filters/transferFilter/getMinMaxTransfersDurationForRoute';

import {isEnabled, disable} from 'utilities/flags/flags';
import {isNighttimeTransfer} from 'projects/avia/lib/isNighttimeTransfer';
import {PriceComparator} from 'utilities/currency/compare';

const {EXCLUDE_NIGHTLY, NO_AIRPORT_CHANGE, NO_TRANSFERS, ONE_CHANGE} =
    EAviaSearchTransferFilter;

export function enreachViewData(
    variant: IResultAviaVariant,
    value: IAviaSearchTransferFilter,
    viewData: TransferFilterViewData,
    filtered: Record<string, IResultAviaVariant>,
    filteredByOthers: Record<string, boolean>,
) {
    if (filteredByOthers[variant.tag] || filtered[variant.tag]) {
        return;
    }

    const durations = getMinMaxTransfersDurationForRoute(variant.route);
    if (durations) {
        viewData.maxDuration = Math.max(durations.max, viewData.maxDuration);
        viewData.minDuration = Math.min(durations.min, viewData.minDuration);
    }

    updateDisabled(value, viewData, variant);
}

function updateDisabled(
    {value}: IAviaSearchTransferFilter,
    viewData: TransferFilterViewData,
    variant: IResultAviaVariant,
) {
    if (isEnabled(viewData.disabled, EXCLUDE_NIGHTLY)) {
        if (
            isEnabled(value, EXCLUDE_NIGHTLY) ||
            variant.route.every(withoutNighttimeTransfers)
        ) {
            viewData.disabled = disable(viewData.disabled, EXCLUDE_NIGHTLY);
        }
    }

    if (isEnabled(viewData.disabled, NO_AIRPORT_CHANGE)) {
        if (
            isEnabled(value, NO_AIRPORT_CHANGE) ||
            variant.route.every(withoutAirportChange)
        ) {
            viewData.disabled = disable(viewData.disabled, NO_AIRPORT_CHANGE);
        }
    }

    const changes = Math.max(...variant.route.map(seg => seg.length - 1));
    if (isEnabled(viewData.disabled, ONE_CHANGE)) {
        if (isEnabled(value, ONE_CHANGE) || changes === 1) {
            viewData.disabled = disable(viewData.disabled, ONE_CHANGE);
        }
    }
    if (isEnabled(viewData.disabled, NO_TRANSFERS)) {
        if (isEnabled(value, NO_TRANSFERS) || changes === 0) {
            viewData.disabled = disable(viewData.disabled, NO_TRANSFERS);
        }
    }
}

function withoutNighttimeTransfers(segment: IResultAviaFlight[]) {
    return segment.length ? !hasNighttimeTransfers(segment) : true;
}

function hasNighttimeTransfers(segment: IResultAviaFlight[]) {
    for (let i = 0; i < segment.length - 1; i++) {
        const arrivalFlight = segment[i];
        const departureFlight = segment[i + 1];
        const departureDate = departureFlight.departure.local;
        const arrivalDate = arrivalFlight.arrival.local;

        if (isNighttimeTransfer({departureDate, arrivalDate})) {
            return true;
        }
    }

    return false;
}

function withoutAirportChange(segment: IResultAviaFlight[]) {
    return segment.length ? !hasAirportChange(segment) : true;
}

function hasAirportChange(segment: IResultAviaFlight[]) {
    for (let i = 1; i < segment.length; ++i) {
        if (segment[i].from.id !== segment[i - 1].to.id) {
            return true;
        }
    }

    return false;
}

export function aviaUpdateTransferViewData(
    variants: IResultAviaVariant[],
    value: IAviaSearchTransferFilter,
    viewData: TransferFilterViewData,
    filtered: Record<string, IResultAviaVariant>,
    filteredByOthers: Record<string, boolean>,
) {
    for (let i = 0; i < variants.length; i++) {
        enreachViewData(
            variants[i],
            value,
            viewData,
            filtered,
            filteredByOthers,
        );
    }

    return viewData;
}

export function aviaUpdateNoTransferMinPrice(
    variants: IResultAviaVariant[],
    filtered: Record<string, IResultAviaVariant>,
    filteredByOthers: Record<string, boolean>,
    {min}: PriceComparator,
) {
    let minPrice = null;

    for (let i = 0; i < variants.length; i++) {
        const variant = variants[i];

        if (
            !filteredByOthers[variant.tag] &&
            !filtered[variant.tag] &&
            variant.route.every(segments => segments.length <= 1)
        ) {
            minPrice = minPrice
                ? min(variant.price.tariff, minPrice)
                : variant.price.tariff;
        }
    }

    return minPrice;
}
