import {times, constant} from 'lodash';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IResultVariantPrice} from 'selectors/avia/utils/denormalization/prices';
import {IAviaVariantGroup} from 'selectors/avia/utils/denormalization/variantGroup';

import {CurrencyType} from 'utilities/currency/CurrencyType';

interface IVariantOptions {
    forwardNumber: string;
    forwardTransfers?: number;
    cost?: number;
    duration?: number;
    backwardNumber?: string;
    backwardTransfers?: number;
    fromCompany?: boolean;
    variants?: IResultAviaVariant[];
}

const DEFAULT_COST = 10000;

export const getVariant = ({
    cost = DEFAULT_COST,
    forwardNumber,
    forwardTransfers = 0,
    backwardNumber,
    backwardTransfers = 0,
    fromCompany = false,
}: IVariantOptions): IResultAviaVariant => {
    const price = {
        fromCompany,
        tariff: {
            value: cost,
            currency: CurrencyType.RUB,
        },
    } as IResultVariantPrice;
    const forward: IResultAviaFlight[] = forwardNumber
        ? times(
              forwardTransfers + 1,
              constant({
                  number: forwardNumber,
              } as IResultAviaFlight),
          )
        : [];
    const backward: IResultAviaFlight[] = backwardNumber
        ? times(
              backwardTransfers + 1,
              constant({
                  number: backwardNumber,
              } as IResultAviaFlight),
          )
        : [];

    return {
        price,
        forward,
        backward,
        forwardNumber,
        backwardNumber,
    } as IResultAviaVariant;
};

export const getGroupVariant = (
    options: IVariantOptions,
): IAviaVariantGroup => {
    const {duration = 745, variants} = options;

    return {
        variants: variants ? variants : [getVariant(options)],
        cheapestVariantsByTariff: {
            cheapest: variants ? variants[0] : getVariant(options),
        },
        flightsTotalDuration: duration,
    } as IAviaVariantGroup;
};

export const BASE_GROUP_VARIANT = getGroupVariant({
    forwardNumber: 'SU 1',
});

export const MIN_PRICE_GROUP_VARIANT = getGroupVariant({
    cost: 5000,
    forwardNumber: 'SU 2',
});

const BASE_VARIANT = getVariant({
    forwardNumber: 'SU 3',
});
const COMPANY_VARIANT = getVariant({
    forwardNumber: 'SU 4',
    fromCompany: true,
});
export const COMPANY_PRICE_GROUP_VARIANT = {
    ...MIN_PRICE_GROUP_VARIANT,
    variants: [BASE_VARIANT, COMPANY_VARIANT],
};
