import {flatten} from 'lodash';

import {
    IBaggageTDTariff,
    IVariantPrice,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {
    emptyFareFamilyKeys,
    IFFBaggageTermRule,
    TFareFamilyKey,
    TVariantFareFamilyKeys,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {TNormalizedFareFamilies} from 'reducers/avia/utils/ticketDaemon/normalizeFareFamilies';
import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {checkBaggage} from 'selectors/avia/utils/denormalization/prices';

import {isBaggageIncluded} from 'projects/avia/lib/baggage';

function worstItemVariant() {
    return {
        source: null,
        count: null,
    };
}

function worstBaggageVariant() {
    return {
        included: worstItemVariant(),
        pc: worstItemVariant(),
        wt: worstItemVariant(),
    };
}

export function noBaggage(val: Boolean, baggageTariff: IBaggageTDTariff) {
    return val || !isBaggageIncluded(baggageTariff);
}

// Получаем багажность из тарифов,
// если данных в тарифах нет - то из информации от партнёра/бд
export function denormalizeBaggage(
    price: IVariantPrice,
    reference: INormalizedTDReference,
): IBaggageTDTariff[][] {
    const {baggageTariffs, fareFamilies} = reference;
    const families = price.fare_families || emptyFareFamilyKeys;

    return families.map((segmentFamily, segmentIndex) =>
        segmentFamily.map((family, legIndex) => {
            if (!family) {
                return baggageTariffs[price.baggage[segmentIndex][legIndex]];
            }

            const familyBaggage = fareFamilies[family]?.terms?.baggage;

            if (familyBaggage) {
                return tariffFromFareFamily(familyBaggage);
            }

            return baggageTariffs.None;
        }),
    );
}

/**
 * Подготовка всех полей связанных с багажем внутри денормализованной цены
 */
export function denormalizeBaggageInfo(
    price: IVariantPrice,
    reference: INormalizedTDReference,
) {
    const baggage = denormalizeBaggage(price, reference);
    const hasBaggage = checkBaggage(flatten(baggage));

    const fareFamiliesKeys = price.fare_families || emptyFareFamilyKeys;
    const worstBaggage: Nullable<IBaggageTDTariff> = hasBaggage
        ? getWorstBaggage(
              fareFamiliesKeys,
              reference.fareFamilies,
              flatten(baggage),
          )
        : null;

    return {
        baggage,
        worstBaggage,
        hasBaggage,
    };
}

// Получаем багажность только из информации от партнёра/бд
export function denormalizePartnersBaggage(
    price: IVariantPrice,
    baggageTariffs: Record<string, IBaggageTDTariff>,
): IBaggageTDTariff[][] {
    return price.baggage.map(
        segmentBaggage =>
            segmentBaggage.map(baggageId => baggageTariffs[baggageId]) ||
            baggageTariffs.None,
    );
}

export function parseAviaBaggage(
    baggage: IBaggageTDTariff[],
): IBaggageTDTariff {
    if (!baggage || !baggage.length) {
        return worstBaggageVariant();
    }

    return baggage.reduce(
        (worstTariff: IBaggageTDTariff, variantTariff: IBaggageTDTariff) => {
            // Тариф не заполнен
            if (!variantTariff) {
                return worstBaggageVariant();
            }

            // Дополняем пустые поля тарифа
            if (
                !variantTariff.included ||
                !variantTariff.pc ||
                !variantTariff.wt
            ) {
                variantTariff.included =
                    variantTariff.included || worstItemVariant();
                variantTariff.pc = variantTariff.pc || worstItemVariant();
                variantTariff.wt = variantTariff.wt || worstItemVariant();
            }

            if (typeof worstTariff.included === 'undefined') {
                return variantTariff;
            }

            if (!worstTariff.included || !worstTariff.pc || !worstTariff.wt) {
                worstTariff.included =
                    worstTariff.included || worstItemVariant();
                worstTariff.pc = worstTariff.pc || worstItemVariant();
                worstTariff.wt = worstTariff.wt || worstItemVariant();
            }

            if (
                Number(variantTariff.included.count) <
                    Number(worstTariff.included.count) ||
                Number(variantTariff.pc.count) < Number(worstTariff.pc.count) ||
                Number(variantTariff.wt.count) < Number(worstTariff.wt.count)
            ) {
                return variantTariff;
            }

            return worstTariff;
        },
        {} as IBaggageTDTariff,
    );
}

export function getWorstBaggage(
    fareFamiliesKeys: TVariantFareFamilyKeys,
    fareFamilies: TNormalizedFareFamilies,
    flatBaggage: IBaggageTDTariff[],
) {
    const familyKeys = flatten(fareFamiliesKeys);
    const worstFareFamilyKey = getWorstFareFamily(
        familyKeys.filter(isString),
        fareFamilies,
    );

    if (worstFareFamilyKey) {
        const {baggage} = fareFamilies[worstFareFamilyKey].terms;

        if (baggage) {
            return tariffFromFareFamily(baggage);
        }
    }

    return parseAviaBaggage(flatBaggage);
}

function getWorstFareFamily(
    familyKeys: TFareFamilyKey[],
    fareFamilies: TNormalizedFareFamilies,
) {
    const defaultBaggage = {
        places: 0,
        size: 0,
        weight: 0,
    };

    if (familyKeys.length) {
        return familyKeys.reduce((worstKey, nextKey) => {
            const currentFamilyTermsBaggage = Object.assign(
                {},
                defaultBaggage,
                fareFamilies[worstKey].terms.baggage,
            );
            const nextFamilyTermsBaggage = Object.assign(
                {},
                defaultBaggage,
                fareFamilies[nextKey].terms.baggage,
            );

            if (
                currentFamilyTermsBaggage.places >
                    nextFamilyTermsBaggage.places ||
                currentFamilyTermsBaggage.size > nextFamilyTermsBaggage.size ||
                currentFamilyTermsBaggage.weight > nextFamilyTermsBaggage.weight
            ) {
                return nextKey;
            }

            return worstKey;
        });
    }

    return null;
}

function isString(key: Nullable<string>): key is string {
    return typeof key === 'string';
}

function tariffFromFareFamily({
    places,
    weight,
}: IFFBaggageTermRule): IBaggageTDTariff {
    return {
        included: {
            count: places > 0 ? 1 : 0,
            source: 'fare_families',
        },
        pc: {
            count: places,
            source: 'fare_families',
        },
        wt: {
            count: weight,
            source: 'fare_families',
        },
    };
}
