import {
    IAviaCompany,
    ICompanyTariff,
    IFlightDate,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';
import {EAviaSearchTimeFilter} from 'reducers/avia/search/results/filters/reducer';

import {
    denormalizeStation,
    IResultAviaStation,
} from 'selectors/avia/utils/denormalization/station';

import {getTimeOfTheDay} from 'projects/avia/lib/search/filters/getTimeOfTheDay';

import {IResultCompany, denormalizeCompany} from './company';

export interface IResultOperating {
    company: IResultCompany;
    number: string;
}

export interface IResultAviaFlight {
    key: string;
    arrival: IFlightDate;
    departure: IFlightDate;
    aviaCompany?: IAviaCompany;
    company: IResultCompany;
    companyTariff?: ICompanyTariff;
    from: IResultAviaStation;
    to: IResultAviaStation;
    number: string;
    tModel: string | null;

    // Поля используются для более быстрых сортировок
    arrivalTime: number;
    departureTime: number;

    // Поля используются для быстрой фильтрации
    arrivalTimeOfDay: EAviaSearchTimeFilter;
    departureTimeOfDay: EAviaSearchTimeFilter;
    operating: IResultOperating | null;
}

export function denormalizeFlight(
    flightId: string,
    reference: INormalizedTDReference,
): IResultAviaFlight | null {
    const {
        arrival,
        aviaCompany,
        company,
        companyTariff,
        departure,
        from,
        key,
        number,
        tModel,
        to,
        arrivalTime,
        departureTime,
        operating,
    } = reference.flights[flightId];

    if (!arrival || !departure) {
        return null;
    }

    const denormalizeOperation = operating
        ? {
              company: operating.company
                  ? denormalizeCompany(
                        reference.companies[operating.company],
                        reference,
                    )
                  : getFakeCompany(),
              number: operating.number,
          }
        : null;

    return {
        key,
        arrival,
        departure,
        aviaCompany: aviaCompany
            ? reference.aviaCompanies[aviaCompany]
            : undefined,
        company: company
            ? denormalizeCompany(reference.companies[company], reference)
            : getFakeCompany(),
        companyTariff: companyTariff
            ? reference.companyTariffs[companyTariff]
            : undefined,
        from: denormalizeStation(from, reference),
        to: denormalizeStation(to, reference),
        number,
        tModel,
        arrivalTime,
        arrivalTimeOfDay: getTimeOfTheDay(arrival.local),
        departureTime,
        departureTimeOfDay: getTimeOfTheDay(departure.local),
        operating: denormalizeOperation,
    };
}

function getFakeCompany(): IResultCompany {
    return {
        id: -1, // Бекендеры обещали, что компании с таким id не будет
        logoSvg: '',
        title: '',
        url: '',
        alliance: null,
        color: '',
    };
}
