/* eslint-disable camelcase */
import values from 'lodash/values';
import flatten from 'lodash/flatten';

import {
    IBaggageTDTariff,
    ITDPartner,
    IVariantPrice,
    TStatus,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {
    IFFRefundableTermRule,
    emptyFareFamilyKeys,
    TVariantFareFamilyKeys,
    IFFChangingCarriageTermRule,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {IAviaCarryOn} from 'types/avia/common/IAviaCarryOn';
import {EAviaClassType} from 'types/avia/EAviaClassType';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {
    denormalizePartnersBaggage,
    denormalizeBaggageInfo,
} from 'selectors/avia/utils/denormalization/baggage';
import {denormalizeRefundInfo} from 'selectors/avia/utils/denormalization/refund';
import denormalizeCarryOn from 'selectors/avia/utils/denormalization/carryOn';
import denormalizeTariffName from 'selectors/avia/utils/denormalization/tariffName';

import IPrice from 'utilities/currency/PriceInterface';
import {isBaggageIncluded} from 'projects/avia/lib/baggage';

import {denormalizeChangingCarriage} from './changingCarriage';

export interface IResultVariantPrice {
    baggage: IBaggageTDTariff[][];
    carryOn: Nullable<IAviaCarryOn>[][];
    hasBaggage: boolean;
    partnersBaggage: IBaggageTDTariff[][];
    hasPartnersBaggage: boolean;
    worstBaggage: Nullable<IBaggageTDTariff>;
    charter?: boolean;
    fromCompany?: boolean;
    partner: ITDPartner;
    queryTime: number;
    tariff: IPrice;
    tariffSign: string;
    status: TStatus;
    fareCodes: string[][][];
    fareFamilies: TVariantFareFamilyKeys;
    fareFamiliesHash: string;
    boy: boolean;
    selfConnect: boolean;
    plusPoints?: number;
    tariffName: string | null;
    worstRefund: IFFRefundableTermRule | null;
    hasFreeRefund: boolean;
    hasChargedRefund: boolean;
    hasNoRefund: boolean;
    /** Класс обслуживания */
    travelClass?: EAviaClassType[];
    /** Обмен */
    changingCarriage?: IFFChangingCarriageTermRule;
}

export function checkBaggage(baggage: IBaggageTDTariff[]): boolean {
    return baggage.length > 0 && baggage.every(isBaggageIncluded);
}

export function denormalizePrices(
    pricesMap: Record<string, IVariantPrice[]>,
    reference: INormalizedTDReference,
): IResultVariantPrice[] {
    const prices = flatten(values(pricesMap));

    return prices.map(price => {
        const {
            charter,
            fromCompany,
            partnerCode,
            queryTime,
            selfconnect,
            tariff,
            tariff_sign,
            boy,
            fare_families,
        } = price;

        const denormalizedCarryOn = denormalizeCarryOn(price, reference);
        const partnersBaggage = denormalizePartnersBaggage(
            price,
            reference.baggageTariffs,
        );
        const hasPartnersBaggage = checkBaggage(flatten(partnersBaggage));

        const fareFamiliesKeys = fare_families || emptyFareFamilyKeys;

        const tariffName = denormalizeTariffName(fare_families, reference);

        const travelClass = fare_families?.reduce<EAviaClassType[]>(
            (result, segment) => {
                segment.forEach(familyKey => {
                    if (familyKey) {
                        const fareFamily = reference.fareFamilies[familyKey];

                        result.push(fareFamily.baseClass);
                    }
                });

                return result;
            },
            [],
        );

        const changingCarriage = denormalizeChangingCarriage(price, reference);

        return {
            carryOn: denormalizedCarryOn,
            partnersBaggage,
            hasPartnersBaggage,
            charter,
            fromCompany,
            partner: reference.partners[partnerCode],
            status: reference.partners[partnerCode].status,
            queryTime,
            tariff,
            tariffSign: tariff_sign,
            fareCodes: price.fare_codes ? price.fare_codes : [[], []],
            fareFamilies: fareFamiliesKeys,
            fareFamiliesHash: price.fare_families_hash,
            boy,
            selfConnect: selfconnect,
            plusPoints: price.plusPoints,
            tariffName,
            travelClass,
            changingCarriage,
            ...denormalizeBaggageInfo(price, reference),
            ...denormalizeRefundInfo(price, reference),
        };
    });
}
