import {
    EFareFamilyTermCode,
    emptyFareFamilyKeys,
    IFFRefundableTermRule,
    EFareFamilyAvailability,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {IVariantPrice} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

/** используем 'возврат не доступен' если для перелета нет FareFamily */
const defaultNotAvailableRefund: IFFRefundableTermRule = {
    availability: EFareFamilyAvailability.notAvailable,
    comment: 'FareFamily not exist',
};

export function denormalizeRefund(
    price: IVariantPrice,
    reference: INormalizedTDReference,
): IFFRefundableTermRule[] {
    const {fareFamilies} = reference;

    const fareFamiliesKeys = price.fare_families || emptyFareFamilyKeys;

    return fareFamiliesKeys.flatMap(segmentFamily => {
        return segmentFamily.reduce<IFFRefundableTermRule[]>((arr, family) => {
            /** если family === null - для соответсвующего перелета нет информации по тарифу */
            const fareFamilyNotExist = family === null;

            if (fareFamilyNotExist) {
                arr.push(defaultNotAvailableRefund);
            } else {
                const familyRefund =
                    fareFamilies[family]?.terms?.[
                        EFareFamilyTermCode.refundable
                    ];

                if (familyRefund) {
                    arr.push(familyRefund);
                }
            }

            return arr;
        }, []);
    });
}

const REFUND_PRIORITY = {
    [EFareFamilyAvailability.notAvailable]: 1,
    [EFareFamilyAvailability.charge]: 2,
    [EFareFamilyAvailability.free]: 3,
};

/** Среди всех вариантов возврата определяем наиболее плохой */
export function getWorstRefund(
    refund: IFFRefundableTermRule[],
): IFFRefundableTermRule | null {
    return refund.length > 0
        ? refund.reduce((worstItem, item) => {
              if (typeof worstItem === 'undefined') {
                  return item;
              }

              const availability = REFUND_PRIORITY[item?.availability] ?? 0;
              const worstAvailability =
                  REFUND_PRIORITY[worstItem?.availability] ?? 0;

              return worstAvailability > availability ? item : worstItem;
          })
        : null;
}

/** Подготовка всех полей связанных с возвратом внутри денормализованной цены */
export function denormalizeRefundInfo(
    price: IVariantPrice,
    reference: INormalizedTDReference,
): {
    worstRefund: IFFRefundableTermRule | null;
    hasFreeRefund: boolean;
    hasChargedRefund: boolean;
    hasNoRefund: boolean;
} {
    const refund = denormalizeRefund(price, reference);
    const worstRefund = getWorstRefund(refund);
    const worstRefundType = worstRefund?.availability;

    return {
        worstRefund,
        hasFreeRefund: worstRefundType === EFareFamilyAvailability.free,
        hasChargedRefund: worstRefundType === EFareFamilyAvailability.charge,
        hasNoRefund: worstRefundType === EFareFamilyAvailability.notAvailable,
    };
}
