import {
    IAviaCompany,
    ICompanyTariff,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import * as i18nBlock from 'i18n/avia-baggage-factory';

enum EAviaTariffInfoItem {
    baggage = 'baggage',
    carryon = 'carryon',
}

interface IAviaTariffInfoItem {
    status: string;
    text: string;
    type: EAviaTariffInfoItem;
}

interface IAviaTariffInfo {
    free: string[];
    paid: string[];
    neutral: string[];
    short: IAviaTariffInfoItem[];
}

interface IBuilderData {
    tariff: ICompanyTariff;
    aviaCompany: IAviaCompany;
    isBusiness: boolean;
}

function getEmptyTariffInfo(): IAviaTariffInfo {
    return {
        free: [],
        paid: [],
        neutral: [],
        short: [],
    };
}

function getCarryonFeatures(tariff: ICompanyTariff, aviaCompany: IAviaCompany) {
    return [
        tariff &&
            tariff.carryonNorm &&
            i18nBlock.weight({
                weight: tariff.carryonNorm,
            }),
        aviaCompany &&
            aviaCompany.carryonWidth &&
            aviaCompany.carryonHeight &&
            aviaCompany.carryonLength &&
            i18nBlock.dimensions({
                width: aviaCompany.carryonWidth,
                height: aviaCompany.carryonHeight,
                length: aviaCompany.carryonLength,
            }),
    ].filter(Boolean);
}

function buildBaggageInfo(
    info: IAviaTariffInfo,
    {tariff, aviaCompany, isBusiness}: IBuilderData,
) {
    const costType = aviaCompany && aviaCompany.costType;

    if (!tariff || !tariff.published || isBusiness) {
        info.neutral.push(i18nBlock.baggageDashCheck());

        if (costType === 'normal' || isBusiness) {
            info.short.push({
                status: 'free',
                text: i18nBlock.baggageDashCheckDashShort(),
                type: EAviaTariffInfoItem.baggage,
            });
        }
    } else if (!tariff.baggageAllowed && costType !== 'lowcost') {
        info.neutral.push(i18nBlock.baggageDashCheck());
    }

    return info;
}

function buildCarryon(
    info: IAviaTariffInfo,
    {tariff, aviaCompany, isBusiness}: IBuilderData,
) {
    if (!tariff || !tariff.published || isBusiness) {
        return info;
    }

    const features = getCarryonFeatures(tariff, aviaCompany);

    if (tariff.carryon) {
        if (features.length) {
            info.free.push(
                i18nBlock.carryonDashUpDashTo() + features.join(', '),
            );
            info.short.push({
                status: 'free',
                text: i18nBlock.carryonDashFreeDashUpDashTo() + features[0],
                type: EAviaTariffInfoItem.carryon,
            });
        } else {
            info.free.push(i18nBlock.carryonDashOne());
            info.short.push({
                status: 'free',
                text: i18nBlock.carryonDashFree(),
                type: EAviaTariffInfoItem.carryon,
            });
        }
    } else {
        info.paid.push(i18nBlock.carryonDashAll());
        info.short.push({
            status: 'paid',
            text: i18nBlock.carryonDashPaid(),
            type: EAviaTariffInfoItem.carryon,
        });
    }

    return info;
}

export function buildTariffInfo(
    companyTariff: ICompanyTariff,
    aviaCompany: IAviaCompany,
    isBusiness: boolean,
): IAviaTariffInfo {
    const emptyTariffInfo = getEmptyTariffInfo();

    if (!companyTariff && !aviaCompany) {
        return emptyTariffInfo;
    }

    return [buildBaggageInfo, buildCarryon].reduce(
        (info, builder) =>
            builder(info, {
                tariff: companyTariff,
                aviaCompany,
                isBusiness,
            }),
        emptyTariffInfo,
    );
}
