import {
    TStatus,
    IBaggageTDTariff,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {INormalizedFare} from 'reducers/avia/utils/ticketDaemon/normalizeTDVariants';
import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {
    denormalizePrices,
    IResultVariantPrice,
} from 'selectors/avia/utils/denormalization/prices';
import {
    denormalizeFlight,
    IResultAviaFlight,
} from 'selectors/avia/utils/denormalization/flight';

import {getVariantId} from 'server/loggers/avia/AviaVariantsLog/utils/getVariantId';

export interface IResultAviaVariant {
    key: string;
    tag: string;
    hybrid?: boolean;
    route: IResultAviaFlight[][];
    price: IResultVariantPrice;

    forward: IResultAviaFlight[];
    backward: IResultAviaFlight[];
    forwardRoute: string;
    backwardRoute: string;
    forwardNumber: string;
    backwardNumber: string | null;

    status: TStatus;
    fromCode: string;
    toCode: string;
    baggage: Nullable<IBaggageTDTariff>;
    variantId: string;

    // Поля нужны для оптимизации фильтрации
    hasBaggage: boolean;
    // Поле служит для построения ссылки на редирект
    // Используем временно (из-за расхождений в багажности полученной от партнёра и из тарифов)
    hasPartnersBaggage: boolean;
}

export function createResultAviaVariants(
    normalizedVariant: INormalizedFare,
    reference: INormalizedTDReference,
): IResultAviaVariant[] | null {
    const {route, prices, hybrid} = normalizedVariant;
    const denormalizedRoute: IResultAviaFlight[][] = [];

    for (const segment of route) {
        const denormalizedSegment = [];

        for (const flightId of segment) {
            const flight = denormalizeFlight(flightId, reference);

            if (!flight) {
                return null;
            }

            denormalizedSegment.push(flight);
        }

        denormalizedRoute.push(denormalizedSegment);
    }

    const denormalizedPrices = denormalizePrices(prices, reference);

    const forward = denormalizedRoute[0];
    const backward = denormalizedRoute[1] || [];
    const forwardRoute = forward.map(getFlightKey).join(',');
    const backwardRoute = backward.map(getFlightKey).join(',');
    const forwardNumbers = forward.map(getFlightNumber);
    const backwardNumbers = backward.map(getFlightNumber);

    const {0: firstFlight, [forward.length - 1]: lastFlight} = forward;

    return denormalizedPrices.map((price): IResultAviaVariant => {
        const tag =
            forwardRoute +
            backwardRoute +
            price.partner.code +
            Number(price.hasBaggage) +
            Number(price.hasChargedRefund) +
            Number(price.hasFreeRefund);

        return {
            key: getVariantKey(forwardRoute, backwardRoute),
            tag,
            hybrid,
            route: denormalizedRoute,
            price,

            forward,
            backward,
            forwardRoute,
            backwardRoute,
            forwardNumber: forwardNumbers[0],
            backwardNumber: backwardNumbers.length ? backwardNumbers[0] : null,

            hasBaggage: price.hasBaggage,
            hasPartnersBaggage: price.hasPartnersBaggage,

            status: reference.partners[price.partner.code].status,
            fromCode: firstFlight.from.code,
            toCode: lastFlight.to.code,
            baggage: price.worstBaggage,
            variantId: getVariantId(price.partner.code),
        };
    });
}

function getFlightKey({departure, arrival, number}: IResultAviaFlight): string {
    const datetime = departure || arrival;
    const datetimePart = (datetime && datetime.local.slice(0, -3)) || '';

    return number + '.' + datetimePart;
}

function getVariantKey(forward: string, backward: string): string {
    return forward + (backward ? ' ' + backward : '');
}

function getFlightNumber({number}: IResultAviaFlight): string {
    return number;
}
