import {last, values} from 'lodash';

import {getVariantOverallDuration} from 'projects/avia/lib/search/getVariantOverallDuration';
import {compareResultVariants} from 'projects/avia/lib/order/comparators/compareResultVariants';
import {ECompareResult} from 'projects/avia/lib/comparators/TComparator';
import {PriceComparator} from 'utilities/currency/compare';

import {IResultAviaVariant} from './variant';
import {IResultAviaFlight} from './flight';

export enum EAviaVariantGroupType {
    /** Удобный перелёт */
    comfy = 'comfy',
    /** Лучшая цена */
    cheapest = 'cheapest',
    /** Популярный рейс */
    popular = 'popular',
    /** Продаёт авиакомпания */
    aviacompany = 'aviacompany',
    /** Предложение с доступным BoY */
    boy = 'boy',
}

export interface ICheapestVariantsByTariff {
    cheapest: IResultAviaVariant;
    cheapestWithBaggage: IResultAviaVariant | null;
    withBaggage: {
        noOrUnknownRefund: IResultAviaVariant | null;
        chargedRefund: IResultAviaVariant | null;
        freeRefund: IResultAviaVariant | null;
    };
    withoutBaggage: {
        noOrUnknownRefund: IResultAviaVariant | null;
        chargedRefund: IResultAviaVariant | null;
        freeRefund: IResultAviaVariant | null;
    };
}

export interface IAviaVariantGroup {
    variants: IResultAviaVariant[];
    cheapestVariantsByTariff: ICheapestVariantsByTariff;
    key: string;
    type?: EAviaVariantGroupType;

    // Поля для оптимизации сортировок
    flightsTotalDuration: number;
    departureLocalTime: string;
    arrivalLocalTime: string;
}

function initVariantGroup(variant: IResultAviaVariant): IAviaVariantGroup {
    return {
        key: variant.key,
        variants: [],
        cheapestVariantsByTariff: {
            cheapest: variant,
            cheapestWithBaggage: null,
            withBaggage: {
                noOrUnknownRefund: null,
                chargedRefund: null,
                freeRefund: null,
            },
            withoutBaggage: {
                noOrUnknownRefund: null,
                chargedRefund: null,
                freeRefund: null,
            },
        },
        flightsTotalDuration: getVariantOverallDuration(variant.route),
        departureLocalTime: getDepartureTime(variant.route),
        arrivalLocalTime: getArrivalTime(variant.route),
    };
}

export function groupAviaVariants(
    variants: IResultAviaVariant[],
    priceComparator: PriceComparator,
): IAviaVariantGroup[] {
    const result = {} as Record<string, IAviaVariantGroup>;
    const compare = compareResultVariants(priceComparator);

    for (const variant of variants) {
        const variantGroup = (result[variant.key] =
            result[variant.key] || initVariantGroup(variant));

        variantGroup.variants.push(variant);

        const {cheapest, cheapestWithBaggage, withBaggage, withoutBaggage} =
            variantGroup.cheapestVariantsByTariff;

        if (
            !cheapest ||
            compare(cheapest.price, variant.price) === ECompareResult.GT
        ) {
            variantGroup.cheapestVariantsByTariff.cheapest = variant;
        }

        if (
            variant.price.hasBaggage &&
            (!cheapestWithBaggage ||
                compare(cheapestWithBaggage.price, variant.price) ===
                    ECompareResult.GT)
        ) {
            variantGroup.cheapestVariantsByTariff.cheapestWithBaggage = variant;
        }

        const baggageSubgroup = variant.price.hasBaggage
            ? withBaggage
            : withoutBaggage;

        if (variant.price.hasFreeRefund) {
            if (
                !baggageSubgroup.freeRefund ||
                compare(baggageSubgroup.freeRefund.price, variant.price) ===
                    ECompareResult.GT
            ) {
                baggageSubgroup.freeRefund = variant;
            }
        } else if (variant.price.hasChargedRefund) {
            if (
                !baggageSubgroup.chargedRefund ||
                compare(baggageSubgroup.chargedRefund.price, variant.price) ===
                    ECompareResult.GT
            ) {
                baggageSubgroup.chargedRefund = variant;
            }
        } else if (
            !baggageSubgroup.noOrUnknownRefund ||
            compare(baggageSubgroup.noOrUnknownRefund.price, variant.price) ===
                ECompareResult.GT
        ) {
            baggageSubgroup.noOrUnknownRefund = variant;
        }
    }

    return values(result);
}

function getDepartureTime(route: IResultAviaFlight[][]): string {
    return route[0][0].departure.local;
}

function getArrivalTime(route: IResultAviaFlight[][]): string {
    return last(route[0])!.arrival.local;
}
