import {createSelector} from 'reselect';

import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';
import EAsyncStatus from 'types/common/EAsyncStatus';

import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import {getHasAviaDynamicsExcludedFilters} from 'selectors/avia/dynamicsPrices/getHasExcludedFilter';
import {getAviaMinSearchPrice} from 'selectors/avia/search/getMinSearchPrice';
import {priceComparatorSelector} from 'selectors/common/currenciesSelector';
import {aviaWeekPricesFinalPricesSelector} from 'selectors/avia/weekPrices/aviaWeekPricesFinalPricesSelector';
import {aviaWeekPricesStatusSelector} from 'selectors/avia/weekPrices/aviaWeekPricesSelectors';
import {aviaWeekPricesRequestParamsSelector} from 'selectors/avia/weekPrices/aviaWeekPricesRequestParamsSelector';
import aviaWeekPricesCurrentRequestParamsSelector from 'selectors/avia/weekPrices/aviaWeekPricesCurrentRequestParamsSelector';

import buildPriceDaysByDates from 'projects/avia/components/Dynamics/utilities/buildPriceDaysByDates';
import dynamicsWeekDates from 'projects/avia/components/Dynamics/utilities/dynamicsWeekDates';
import {
    getPriceHintInfo,
    IDayWithPrice,
} from 'projects/avia/components/Dynamics/utilities/getPriceHintInfo';
import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import {priceSubtract} from 'utilities/currency/priceMath';
import IPrice from 'utilities/currency/PriceInterface';
import isValidDataForRequestDynamic from 'projects/avia/lib/dynamic/isValidDataForRequestDynamic';
import isCompatibleDynamicSearch from 'projects/avia/lib/dynamic/isCompatibleDynamicSearch';
import needRequest from 'projects/avia/lib/dynamic/needRequest';
import getActualStatus from 'projects/avia/lib/dynamic/getActualStatus';

interface IAviaWeekPriceInfo {
    isValidRequestParams: boolean;
    isActual: boolean;
    actualStatus: EAsyncStatus;
    needRequest: boolean;
    priceType: EPriceHintType;

    requestParams?: TValidDataForRequestDynamic;
    weekPrices?: TDynamicsDaysInfo;
    priceDiff?: IPrice;
    minPrice?: IPrice;
    priceDays?: IDayWithPrice[];
}

export default createSelector(
    aviaWeekPricesRequestParamsSelector,
    aviaWeekPricesCurrentRequestParamsSelector,
    priceComparatorSelector,
    aviaWeekPricesFinalPricesSelector,
    getAviaMinSearchPrice,
    getHasAviaDynamicsExcludedFilters,
    aviaWeekPricesStatusSelector,
    (
        requestParams,
        currentRequestParams,
        priceComparator,
        weekPrices,
        minSearchPrice,
        hasExcludedFilter,
        status,
    ): IAviaWeekPriceInfo => {
        let isValidRequestParams: boolean;
        let isActual: boolean;
        let priceType = EPriceHintType.EMPTY;

        if (!requestParams || !isValidDataForRequestDynamic(requestParams)) {
            isValidRequestParams = false;
            isActual = false;

            return {
                isValidRequestParams,
                isActual,
                priceType,
                needRequest: needRequest({
                    isActual,
                    isValidRequestParams,
                }),
                actualStatus: getActualStatus({
                    isActual,
                    status,
                }),
            };
        }

        isValidRequestParams = true;

        if (
            !currentRequestParams ||
            !isCompatibleDynamicSearch(requestParams, currentRequestParams)
        ) {
            isActual = false;

            return {
                isValidRequestParams,
                isActual,
                requestParams,
                priceType,
                needRequest: needRequest({
                    isActual,
                    isValidRequestParams,
                }),
                actualStatus: getActualStatus({
                    isActual,
                    status,
                }),
            };
        }

        isActual = true;

        if (status !== EAsyncStatus.SUCCESS) {
            return {
                isValidRequestParams,
                isActual,
                requestParams,
                priceType,
                needRequest: needRequest({
                    isActual,
                    isValidRequestParams,
                }),
                actualStatus: getActualStatus({
                    isActual,
                    status,
                }),
            };
        }

        const {searchForm} = requestParams;
        const {when} = searchForm;
        const dates = dynamicsWeekDates(when);
        const priceDays = buildPriceDaysByDates(dates, weekPrices);
        const priceHintInfo = getPriceHintInfo(
            priceDays,
            priceComparator,
            minSearchPrice,
        );
        const {otherPrice, minPrice} = priceHintInfo;

        priceType = priceHintInfo.type;

        let priceDiff;

        if (hasExcludedFilter) {
            priceType = EPriceHintType.EMPTY;
        } else {
            if (
                priceType === EPriceHintType.MINIMAL &&
                minSearchPrice &&
                otherPrice
            ) {
                priceDiff = priceSubtract(otherPrice, minSearchPrice);
            }

            if (
                priceType === EPriceHintType.EXPENSIVE &&
                minSearchPrice &&
                minPrice
            ) {
                priceDiff = priceSubtract(minSearchPrice, minPrice);
            }
        }

        return {
            isActual,
            isValidRequestParams,
            needRequest: needRequest({
                isActual,
                isValidRequestParams,
            }),
            actualStatus: getActualStatus({
                isActual,
                status,
            }),
            requestParams,
            priceType,
            priceDiff,
            minPrice,
            weekPrices,
            priceDays,
        };
    },
);
