import {createSelector} from 'reselect';

import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';
import {IBusesFilters} from 'types/buses/search/filters/IBusesFilters';

import {StoreInterface} from 'reducers/storeTypes';

import getPriceFilterRange from 'selectors/buses/search/utilities/filters/getPriceFilterRange';
import getTimeOfDayFilterOptions from 'selectors/buses/search/utilities/filters/getTimeOfDayFilterOptions';
import getStationFilterOptions from 'selectors/buses/search/utilities/filters/getStationFilterOptions';
import getHiddenSegmentsIdsByFilter from 'selectors/buses/search/utilities/filters/getHiddenSegmentsIdsByFilter';
import getAvailableFilterOptions from 'selectors/buses/search/utilities/filters/getAvailableFilterOptions';

export interface IBusesSearchFiltersSelector {
    filters: IBusesFilters;
    hiddenSegmentsIds: Set<string>;
}

export default createSelector(
    (state: StoreInterface) => state.buses.search.segments,
    (state: StoreInterface) => state.buses.search.filters,
    (segmentsInfo, filters): IBusesSearchFiltersSelector | null => {
        const {value: segments} = segmentsInfo;

        if (!segments?.length) {
            return null;
        }

        const {
            price: priceValue,
            departureTime: departureTimeValue,
            arrivalTime: arrivalTimeValue,
            departureStation: departureStationValue,
            arrivalStation: arrivalStationValue,
        } = filters;

        const priceRange = getPriceFilterRange(segments);
        const departureTimeOptions = getTimeOfDayFilterOptions(
            segments,
            EBusesFilter.DEPARTURE_TIME,
        );
        const arrivalTimeOptions = getTimeOfDayFilterOptions(
            segments,
            EBusesFilter.ARRIVAL_TIME,
        );
        const departureStationOptions = getStationFilterOptions(
            segments,
            EBusesFilter.DEPARTURE_STATION,
        );
        const arrivalStationOptions = getStationFilterOptions(
            segments,
            EBusesFilter.ARRIVAL_STATION,
        );

        const hiddenSegmentsIdsByPriceFilter = getHiddenSegmentsIdsByFilter(
            segments,
            {type: EBusesFilter.PRICE, values: priceValue},
        );
        const hiddenSegmentsIdsByDepartureTimeFilter =
            getHiddenSegmentsIdsByFilter(segments, {
                type: EBusesFilter.DEPARTURE_TIME,
                values: departureTimeValue,
            });
        const hiddenSegmentsIdsByArrivalTimeFilter =
            getHiddenSegmentsIdsByFilter(segments, {
                type: EBusesFilter.ARRIVAL_TIME,
                values: arrivalTimeValue,
            });
        const hiddenSegmentsIdsByDepartureStationFilter =
            getHiddenSegmentsIdsByFilter(segments, {
                type: EBusesFilter.DEPARTURE_STATION,
                values: departureStationValue,
            });
        const hiddenSegmentsIdsByArrivalStationFilter =
            getHiddenSegmentsIdsByFilter(segments, {
                type: EBusesFilter.ARRIVAL_STATION,
                values: arrivalStationValue,
            });

        const hiddenSegmentsIds = new Set<string>([
            ...hiddenSegmentsIdsByPriceFilter,
            ...hiddenSegmentsIdsByDepartureTimeFilter,
            ...hiddenSegmentsIdsByArrivalTimeFilter,
            ...hiddenSegmentsIdsByDepartureStationFilter,
            ...hiddenSegmentsIdsByArrivalStationFilter,
        ]);

        const availableDepartureTimeOptions = getAvailableFilterOptions({
            segments,
            type: EBusesFilter.DEPARTURE_TIME,
            options: departureTimeOptions,
            hiddenSegmentsIdsWithoutThisFilter: new Set<string>([
                ...hiddenSegmentsIdsByPriceFilter,
                ...hiddenSegmentsIdsByArrivalTimeFilter,
                ...hiddenSegmentsIdsByDepartureStationFilter,
                ...hiddenSegmentsIdsByArrivalStationFilter,
            ]),
        });

        const availableArrivalTimeOptions = getAvailableFilterOptions({
            segments,
            type: EBusesFilter.ARRIVAL_TIME,
            options: arrivalTimeOptions,
            hiddenSegmentsIdsWithoutThisFilter: new Set<string>([
                ...hiddenSegmentsIdsByPriceFilter,
                ...hiddenSegmentsIdsByDepartureTimeFilter,
                ...hiddenSegmentsIdsByDepartureStationFilter,
                ...hiddenSegmentsIdsByArrivalStationFilter,
            ]),
        });

        const availableDepartureStationOptions = getAvailableFilterOptions({
            segments,
            type: EBusesFilter.DEPARTURE_STATION,
            options: departureStationOptions,
            hiddenSegmentsIdsWithoutThisFilter: new Set<string>([
                ...hiddenSegmentsIdsByPriceFilter,
                ...hiddenSegmentsIdsByDepartureTimeFilter,
                ...hiddenSegmentsIdsByArrivalTimeFilter,
                ...hiddenSegmentsIdsByArrivalStationFilter,
            ]),
        });

        const availableArrivalStationOptions = getAvailableFilterOptions({
            segments,
            type: EBusesFilter.ARRIVAL_STATION,
            options: arrivalStationOptions,
            hiddenSegmentsIdsWithoutThisFilter: new Set<string>([
                ...hiddenSegmentsIdsByPriceFilter,
                ...hiddenSegmentsIdsByDepartureTimeFilter,
                ...hiddenSegmentsIdsByArrivalTimeFilter,
                ...hiddenSegmentsIdsByDepartureStationFilter,
            ]),
        });

        return {
            filters: {
                price: {
                    value: priceValue,
                    range: priceRange,
                    hiddenByFilter: hiddenSegmentsIdsByPriceFilter,
                },
                departureTime: {
                    value: departureTimeValue,
                    options: departureTimeOptions,
                    hiddenByFilter: hiddenSegmentsIdsByDepartureTimeFilter,
                    availableOptions: availableDepartureTimeOptions,
                },
                arrivalTime: {
                    value: arrivalTimeValue,
                    options: arrivalTimeOptions,
                    hiddenByFilter: hiddenSegmentsIdsByArrivalTimeFilter,
                    availableOptions: availableArrivalTimeOptions,
                },
                departureStation: {
                    value: departureStationValue,
                    options: departureStationOptions,
                    hiddenByFilter: hiddenSegmentsIdsByDepartureStationFilter,
                    availableOptions: availableDepartureStationOptions,
                },
                arrivalStation: {
                    value: arrivalStationValue,
                    options: arrivalStationOptions,
                    hiddenByFilter: hiddenSegmentsIdsByArrivalStationFilter,
                    availableOptions: availableArrivalStationOptions,
                },
            },
            hiddenSegmentsIds,
        };
    },
);
