import {createSelector} from 'reselect';

import {IBaseSuggestItem} from 'types/hotels/common/ISearchSuggest';
import {TExperiments} from 'server/providers/experiments/types';

import experimentsSelector from 'selectors/common/experimentsSelector';

import {DateLikeType} from 'utilities/dateUtils';
import humanizePeriod from 'utilities/dateUtils/humanizePeriod';
import {
    formatCountOfAdults,
    formatCountOfChildren,
    formatCountOfGuests,
} from 'utilities/formatCount';
import {getTotalNights} from 'projects/hotels/utilities/calculateTotalNights/calculateTotalNights';

import * as i18nBlock from 'i18n/hotels-SearchInformationSection';

import {
    getSearchInformation,
    GetSearchInformationType,
} from './searchInformationSelector';

const DEFAULT_NIGHTS_COUNT = 1;

export interface IHotelsTextSearchInformation {
    nights: number;
    dates?: string;
    adults: string;
    children?: string;
    guests?: string;
    geoName: string;
    hotelName: string;
    activeSuggest?: IBaseSuggestItem | null;
    experiments: TExperiments;
}

/* Helpers */
export const getNights = (
    checkinDate: DateLikeType,
    checkoutDate: DateLikeType,
): IHotelsTextSearchInformation['nights'] => {
    if (checkinDate && checkoutDate) {
        return getTotalNights(checkinDate, checkoutDate);
    }

    return DEFAULT_NIGHTS_COUNT;
};

export const getDates = (
    checkinDate: DateLikeType,
    checkoutDate: DateLikeType,
): IHotelsTextSearchInformation['dates'] => {
    if (checkinDate && checkoutDate) {
        return humanizePeriod(checkinDate, checkoutDate, {short: true});
    }
};

export const getChildrenInfo = (
    childrenAges: number[],
): IHotelsTextSearchInformation['children'] => {
    if (childrenAges && childrenAges.length > 0) {
        return formatCountOfChildren(childrenAges.length);
    }
};

export const getCountOfGuests = ({
    adults,
    childrenAges,
}: {
    adults: number;
    childrenAges: number[];
}): IHotelsTextSearchInformation['guests'] => {
    const childrenTotal = childrenAges ? childrenAges.length : 0;

    return formatCountOfGuests(adults + childrenTotal);
};

export const textSearchInformationSelector = createSelector(
    getSearchInformation,
    experimentsSelector,
    (
        searchInformation: ReturnType<GetSearchInformationType>,
        experiments,
    ): IHotelsTextSearchInformation => {
        const {
            checkinDate,
            checkoutDate,
            adults,
            childrenAges,
            geoObject: {name: geoName},
            hotel: {name: hotelName},
            activeSuggest,
        } = searchInformation;

        const hasDates = checkinDate && checkoutDate;

        return {
            geoName,
            hotelName,
            nights: getNights(checkinDate, checkoutDate),
            dates: hasDates
                ? getDates(checkinDate, checkoutDate)
                : i18nBlock.dateNotSpecified(),
            adults: hasDates ? formatCountOfAdults(adults) : '',
            children: getChildrenInfo(childrenAges),
            guests: getCountOfGuests({adults, childrenAges}),
            activeSuggest,
            experiments,
        };
    },
);
