import {ITrainsCheapestVariantsInfo} from 'types/trains/search/badgesInfo/ITrainsSearchBadgesInfo';
import {
    ITrainsVariant,
    TTrainsVariantId,
    ITrainsVariantsAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';
import {EDirection} from 'types/common/EDirection';

import IPrice from 'utilities/currency/PriceInterface';
import {getTrainsVariantMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantMinPrice';
import {getTrainsVariantsMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMinPrice';
import {getTrainsVariantsWithPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsWithPrice';

export const RATION_TO_OFF_BADGES = 0.2;
export const MAX_EACH_BADGE_COUNT = 3;

export const DEFAULT_CHEAPEST_VARIANTS_INFO: ITrainsCheapestVariantsInfo = {
    variantIds: [],
    canUseInfo: false,
};

const checkUseCheapestVariantsInfo = ({
    variantsWithPrice,
    variantIds,
}: {
    variantsWithPrice: ITrainsVariant[];
    variantIds: TTrainsVariantId[];
}): boolean => {
    const variantIdsCount = variantIds.length;

    return (
        variantIdsCount < MAX_EACH_BADGE_COUNT &&
        variantIdsCount / variantsWithPrice.length < RATION_TO_OFF_BADGES
    );
};

const filterAndMapVariantIds = ({
    direction,
    variantsWithPrice,
    variantsMinPrice,
}: {
    direction?: EDirection | null;
    variantsWithPrice: ITrainsVariant[];
    variantsMinPrice: IPrice;
}): TTrainsVariantId[] => {
    return variantsWithPrice
        .filter(variant => {
            const variantMinPrice = getTrainsVariantMinPrice({
                variant,
                direction,
            });

            if (!variantMinPrice) {
                return false;
            }

            return variantMinPrice.value === variantsMinPrice.value;
        })
        .map(variant => variant.id);
};

export const getCheapestVariantsInfo = (
    variantsAndDirection: ITrainsVariantsAndDirection,
): ITrainsCheapestVariantsInfo => {
    const {direction} = variantsAndDirection;
    const variantsWithPrice = getTrainsVariantsWithPrice(variantsAndDirection);

    if (!variantsWithPrice?.length) {
        return DEFAULT_CHEAPEST_VARIANTS_INFO;
    }

    const variantsMinPrice = getTrainsVariantsMinPrice({
        variants: variantsWithPrice,
        direction,
    });

    if (!variantsMinPrice) {
        return DEFAULT_CHEAPEST_VARIANTS_INFO;
    }

    const variantIds = filterAndMapVariantIds({
        direction,
        variantsMinPrice,
        variantsWithPrice,
    });

    return {
        variantIds,
        canUseInfo: checkUseCheapestVariantsInfo({
            variantIds,
            variantsWithPrice,
        }),
    };
};
