import {
    ITrainsCheapestVariantsInfo,
    ITrainsFastestVariantsInfo,
} from 'types/trains/search/badgesInfo/ITrainsSearchBadgesInfo';
import {
    ITrainsVariant,
    TTrainsVariantId,
    ITrainsVariantsAndDirection,
} from 'types/trains/common/variant/ITrainsVariant';

import {getTrainsVariantDuration} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantDuration';
import {getTrainsVariantsMinDuration} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantsMinDuration';

export const RATION_TO_OFF_BADGES = 0.2;
export const MAX_EACH_BADGE_COUNT = 3;

export const DEFAULT_FASTEST_VARIANTS_INFO: ITrainsCheapestVariantsInfo = {
    variantIds: [],
    canUseInfo: false,
};

const checkUseFastestVariantsInfo = ({
    variants,
    variantIds,
}: {
    variants: ITrainsVariant[];
    variantIds: TTrainsVariantId[];
}): boolean => {
    const variantIdsCount = variantIds.length;

    return (
        variantIdsCount < MAX_EACH_BADGE_COUNT &&
        variantIdsCount / variants.length < RATION_TO_OFF_BADGES
    );
};

const filterAndMapVariantIds = ({
    variantsAndDirection,
    minDuration,
}: {
    variantsAndDirection: ITrainsVariantsAndDirection;
    minDuration: number;
}): TTrainsVariantId[] => {
    const {variants, direction} = variantsAndDirection;

    return variants
        .filter(variant => {
            const variantDuration = getTrainsVariantDuration({
                variant,
                direction,
            });

            return variantDuration === minDuration;
        })
        .map(variant => variant.id);
};

export const getFastestVariantsInfo = (
    variantsAndDirection: ITrainsVariantsAndDirection,
): ITrainsFastestVariantsInfo => {
    const {variants} = variantsAndDirection;

    if (!variants?.length) {
        return DEFAULT_FASTEST_VARIANTS_INFO;
    }

    const minDuration = getTrainsVariantsMinDuration(variantsAndDirection);

    if (!minDuration) {
        return DEFAULT_FASTEST_VARIANTS_INFO;
    }

    const variantIds = filterAndMapVariantIds({
        variantsAndDirection,
        minDuration,
    });

    return {
        variantIds,
        canUseInfo: checkUseFastestVariantsInfo({variants, variantIds}),
    };
};
