import {createSelector} from 'reselect';

import {ITrainsFilters} from 'types/trains/search/filters/ITrainsFilters';
import {
    ITrainsVariant,
    TTrainsVariantId,
    ITrainsFilteredVariantsInfo,
    TTrainsVariantWithVisibleStatusById,
} from 'types/trains/common/variant/ITrainsVariant';
import {TrainsSearchContextType} from 'reducers/trains/context/types';

import {StoreInterface} from 'reducers/storeTypes';
import {ITrainsSearchReducer} from 'reducers/trains/genericSearch/search/reducer';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import {trainsSortVariants} from 'selectors/trains/genericSearch/sort/trainsSortVaraints';
import {getTrainsSearchSortInfo} from 'selectors/trains/genericSearch/sort/trainsSearchSortInfoSelector';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';
import {getTrainsOriginalSearchFilters} from 'selectors/trains/genericSearch/filters/getTrainsOriginalSearchFilters';

import {getTrainsFilteredVariantsInfo} from 'projects/trains/lib/genericSearch/filters/getTrainsFilteredVariantsInfo';

export interface ITrainsSearchInfoSelector extends ITrainsSearchReducer {}

/*
 * Filter Variant Selectors
 * */

export const trainsOriginalVariantWithVisibleStatusByIdSelector =
    createSelector<
        StoreInterface,
        ITrainsSearchReducer,
        TTrainsVariantWithVisibleStatusById
    >(getTrainsOriginalSearchInfo, originalSearchInfo => {
        const {variants} = originalSearchInfo;

        return variants.reduce<TTrainsVariantWithVisibleStatusById>(
            (filteredVariantsById, variant) => {
                filteredVariantsById[variant.id] = {
                    variant,
                    isVisible: true,
                };

                return filteredVariantsById;
            },
            {},
        );
    });

export const trainsFilteredVariantsInfoSelector = createSelector<
    StoreInterface,
    TrainsSearchContextType,
    ITrainsSearchReducer,
    ITrainsFilters | null,
    TTrainsVariantWithVisibleStatusById,
    ITrainsFilteredVariantsInfo
>(
    trainsContextSelector,
    getTrainsOriginalSearchInfo,
    getTrainsOriginalSearchFilters,
    trainsOriginalVariantWithVisibleStatusByIdSelector,
    (
        context,
        originalSearchInfo,
        filters,
        originalVariantWithVisibleStatusById,
    ): ITrainsFilteredVariantsInfo => {
        const {variants} = originalSearchInfo;

        return getTrainsFilteredVariantsInfo({
            context,
            filters,
            variants,
            originalVariantWithVisibleStatusById,
        });
    },
);

export const trainsFilteredVariantsSelector = createSelector<
    StoreInterface,
    ITrainsFilteredVariantsInfo,
    ITrainsVariant[]
>(
    trainsFilteredVariantsInfoSelector,
    ({variantWithVisibleStatusById}): ITrainsVariant[] => {
        return Object.values(variantWithVisibleStatusById)
            .filter(({isVisible}) => isVisible)
            .map(({variant}) => variant);
    },
);

/*
 * Sort Variant Selectors
 * */

export const trainsSortedVariantIdsSelector = createSelector(
    trainsFilteredVariantsSelector,
    getTrainsSearchSortInfo,
    trainsContextSelector,
    (variants, sortInfo, context): TTrainsVariantId[] => {
        return trainsSortVariants({
            sortInfo,
            variants,
            context,
        }).map(variant => variant.id);
    },
);
